/******************************************************************

  Module: TiCamera.cpp

  Author: Bijan Forutanpour

  Description:   Game camera.

  NOTES: Camera impelementations
     a) stationary camera
     b) stationary panning camera, following player
     c) stationary panning camera, lookat player  (and follow at
        fixed distance)
     d) fixed angle player camera
     e) fixed angle player camera - keyframe animated offset
     f) keyframe animated

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*******************************************************************/
#include "TiCamera.h"

// Aniamtin data is created at 30 frames / 1000 millisecs 
#define MILLISECONDS_TO_FRAMES  .03  

TiCamera::TiCamera()
{
  mCamType=0;
  mCamTransition=0;
  mZoneClock=0;
  mUseMinMax=0;
  mMinDist=0;
  mMaxDist=0;
  mUseRail=0;
  mPlayOnce=0;
  mAnimPlayed=0;
}


/******************************************************************
 * GetOffestPos - Get the xyz offset of the camera from the player.
 * The offset is most probably keframed, so evaluate the anim curve.
 *******************************************************************/
void TiCamera::GetOffsetPos (float fFrame, SyVect3* result)
{
  float tKeyStartFrame, tKeyEndFrame;
  tKeyStartFrame = mPlayerOffsetX.GetKeyTime (0);
  tKeyEndFrame =   mPlayerOffsetX.GetKeyTime (mPlayerOffsetX.GetNumKeys()- 1);


  if (fFrame < tKeyStartFrame) 
  {
    fFrame = tKeyStartFrame;
  }
  if (fFrame > tKeyEndFrame) 
  {
    fFrame = tKeyEndFrame;
  }

  result->X = mPlayerOffsetX.GetValue (fFrame);
  result->Y = mPlayerOffsetY.GetValue (fFrame);
  result->Z = mPlayerOffsetZ.GetValue (fFrame);
}



/*******************************************************************
 * GetOffestPos - Get the xyz offset of the camera's interestpoint 
 * from the player. The offset is most probably keframed, so evaluate 
 * the anim curve.
 *******************************************************************/
void TiCamera::GetOffsetInterest (float fFrame, SyVect3* result)
{
  float tKeyStartFrame, tKeyEndFrame;
  tKeyStartFrame = mPlayerOffsetInterestX.GetKeyTime (0);
  tKeyEndFrame =   mPlayerOffsetInterestX.GetKeyTime (mPlayerOffsetInterestX.GetNumKeys()- 1);

  if (fFrame < tKeyStartFrame) 
  {
    fFrame = tKeyStartFrame;
  }
  if (fFrame > tKeyEndFrame) 
  {
    fFrame = tKeyEndFrame;
  }
  
  result->X = mPlayerOffsetInterestX.GetValue (fFrame);
  result->Y = mPlayerOffsetInterestY.GetValue (fFrame);
  result->Z = mPlayerOffsetInterestZ.GetValue (fFrame);
}

/**********************************************************
 * GetOffsetPos - time based, milliseconds. We convert to 
 * frames and use the functions above to sample the animation
 * curve data.
 **********************************************************/
void TiCamera::GetOffsetPos (SyTime animTime, SyVect3* result)
{
  float tKeyStartFrame, tKeyEndFrame;
  float32 animFrame;
  SyTime  animationTimeElapsed;

  /*
   * Get the animation bounds
   */
  tKeyStartFrame = mPlayerOffsetX.GetKeyTime (0);
  tKeyEndFrame =   mPlayerOffsetX.GetKeyTime (mPlayerOffsetX.GetNumKeys()- 1);

  /*
   * Figure out what frame in the animation we want
   */
  animationTimeElapsed = animTime - GetStartTime();
  animFrame = animationTimeElapsed * MILLISECONDS_TO_FRAMES;

  /*
   * Clamp the animation frame
   */
  if (animFrame < tKeyStartFrame) 
  {
    animFrame = tKeyStartFrame;
  }
  if (animFrame > tKeyEndFrame) 
  {
    animFrame = tKeyEndFrame;
  }

  /*
   * Sample the animation curve, return the result
   */
  result->X = mPlayerOffsetX.GetValue (animFrame);
  result->Y = mPlayerOffsetY.GetValue (animFrame);
  result->Z = mPlayerOffsetZ.GetValue (animFrame);
}



/**********************************************************
 * GetOffsetInterest - time based, milliseconds. We convert to 
 * frames and use the functions above to sample the animation
 * curve data.
 **********************************************************/
void TiCamera::GetOffsetInterest (SyTime animTime, SyVect3* result)
{
  float32 animFrame;
  SyTime  animationTimeElapsed;
  float tKeyStartFrame, tKeyEndFrame;

  /*
   * Get the animation bounds
   */
  tKeyStartFrame = mPlayerOffsetX.GetKeyTime (0);
  tKeyEndFrame =   mPlayerOffsetX.GetKeyTime (mPlayerOffsetX.GetNumKeys()- 1);

  /*
   * Figure out what frame in the animation we want
   */
  animationTimeElapsed = animTime - GetStartTime();
  animFrame = animationTimeElapsed * MILLISECONDS_TO_FRAMES;

  /*
   * Clamp the animation frame
   */
  if (animFrame < tKeyStartFrame) 
  {
    animFrame = tKeyStartFrame;
  }
  if (animFrame > tKeyEndFrame) 
  {
    animFrame = tKeyEndFrame;
  }

  /*
   * Sample the animation curve, return the result
   */
  result->X = mPlayerOffsetInterestX.GetValue (animFrame);
  result->Y = mPlayerOffsetInterestY.GetValue (animFrame);
  result->Z = mPlayerOffsetInterestZ.GetValue (animFrame);
}
