/******************************************************************

  Module: TitanI.h

  Author: 

  Description:

  Titan interface.

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*******************************************************************/

#ifndef __TITANI_H__
#define	__TITANI_H__

/*
**
**	 Includes
**
*/

#include "SyTypes.h"

/**/

/*
**
**	 Forward Declarations
**
*/

class SyScene;
class SyPerformanceStats;
namespace PeeringNetwork { class PeeringServerApi; }

/**/

/*
**
**	 Type Definitions
**
*/

//==============================================================================
// Titan->TitanUI callback interface
//==============================================================================

enum TitanButton
{
  ButtonAttackL,
  ButtonAttackS,
  ButtonAction,
  ButtonBlock,
  ButtonAttackRanged,
  ButtonMagic,
  ButtonDodge,
  ButtonToggleAutomap,
  ButtonJump,
  ButtonPause,

  // 
  ButtonMax
};

class TitanControllerI
{
public:
  // Destroys an instance of the TitanControllerI interface.
  virtual ~TitanControllerI() {}

  // returns true if the controller is connected.
  virtual bool GetConnected() const = 0;

  // returns true if the specified button is depressed.
  virtual bool GetButtonState( TitanButton tb ) const = 0;

  // returns a value in the range [-10000,+10000] that indicates whether
  // the directional control is pushed completely forward (+10000),
  // int backward (-10000) or somewhere in between.
  virtual int GetForward() const = 0;

  // returns a value in the range [-10000,+10000] that indicates whether
  // the directional control is pushed completely to the right (+10000),
  // completely to the left (-10000) or somewhere in between.
  virtual int GetRight() const = 0;

  // returns a value in the range [-10000,+10000] that indicates whether
  // the right directional control is pushed completely forward (+10000),
  // int backward (-10000) or somewhere in between.
  virtual int GetRightStickForward() const = 0;

  // returns a value in the range [-10000,+10000] that indicates whether
  // the right directional control is pushed completely to the right (+10000),
  // completely to the left (-10000) or somewhere in between.
  virtual int GetRightStickRight() const = 0;

  static const int DIRECTION_MAX =  10000;

};

class TitanUII
{
public:
  // Destroys an instance of the TitanControllerI interface.
  virtual ~TitanUII() {}

  // Use this method instead of Scene::Clear to give the interface an 
  // opportunity to reload its assets after the clear. 
  virtual int ClearScene() = 0;

  // Get the scene titan can use for drawing
  virtual SyScene* GetScene() = 0;
  virtual SyPerformanceStats* GetPerformanceStats() = 0;

  // Get a pointer to an interface that can be used to poll a controller state
  virtual TitanControllerI* GetController( int controllerId ) = 0;


  // opens up a pawn debug UI
  virtual int  PawnDebugUI(const char *src, int linenum,const char *variables) = 0;
  virtual void PawnDebugOutput(const char *output){};

};

//==============================================================================
// TitanUI->Titan interface
//==============================================================================

class TitanInputHandlerI
{
public:
  // handle controller button state changes
  virtual void OnButtonDown( TitanButton tb ) = 0;
  virtual void OnButtonUp( TitanButton tb ) = 0;
};

class TitanI
{
public:

  // Destroys an instance of the TitanI interface.
  virtual ~TitanI(void) {}

  // Allocates and constructs an instance of the TitanI interface.
  static TitanI* Create( TitanUII* pTitanUI );

  // Allocates resources required by the Titan object.
  virtual bool Init( uint64 time ) = 0;

  // Releases resources acquired by the titan object.
  virtual void Close() = 0;

  // Updates the game state to the specified time.
  virtual void Update( uint64 time ) = 0;

  // Renders the scene.
  // Scene::BeginScene must be called before this method is invoked.
  // Scene::EndScene must be called after this method is invoked.
  virtual void Render() = 0;

  virtual TitanInputHandlerI* GetInputHandler( int controllerId ) = 0;

  // use the specified file to construct all the initial game objects
  virtual void LoadLevel(const char *Filename) = 0;

  // use the specified saved-game file to construct all the initial game object
  virtual void LoadSavedGame(const char *saveGameFilename) = 0;


  // restarts the current level and resets the player
  virtual void Restart() = 0;

  // tells it to join the specified existing network game
  // after network game is joined, you don't need to do anything else as all the initial state
  // will come from the peers in the game you are joining
  virtual void NetworkGameJoin(/*gift*/ PeeringNetwork::PeeringServerApi *api, int gameId) = 0;

  // tell it to create a new network game (and join it)
  // after the network game is created, you must call LoadXml or LoadSavedGame to populated the newly created game
  virtual void NetworkGameCreate(/*gift*/ PeeringNetwork::PeeringServerApi *api, const char *gameName) = 0;

  // this is a temporary hack function to startup a network game until the front-end is completed
  // once the front-end is completed, it will create the PeeringServerApi, do stuff with it, and then
  // pass it off via the NetworkGameJoin or NetworkGameCreate calls
  // This function will join the game with the specified name if it exists, otherwise it will create a game
  // with the specified name using the xmlFilename to setup the initial state of the world
  virtual void HackStartupNetwork(const char *playerName, const char *gameName, const char *xmlFilename, const char *recordFilename, const char *playbackFilename) = 0;

  virtual void ScriptEnableDebugging(){};
  virtual void CheatKey(int key) = 0;

  //============================================================================
  // Design Settings
  //============================================================================

  // Sets whether to log combat events and calculations to a file.
  virtual void SetLogMode(bool on)=0;
  virtual bool GetLogMode()=0;

  // Sets whether to log combat events and calculations to a file.
  virtual void SetCheckArt(bool on){};
  virtual bool GetCheckArt(){return false;};

  // Sets whether player is invulnerable or not
  virtual void SetGodMode(bool on)=0;
  virtual bool GetGodMode()=0;

  // Sets whether to log combat events and calculations to a file.
  virtual void SetInvisibleMode(bool on)=0;
  virtual bool GetInvisibleMode()=0;

  // set whether or not game is paused...
  virtual void PauseGame()=0;
  virtual void UnpauseGame()=0;

  // set time scale (for slow mo mode)
  virtual void SetTimeScale(float scale) = 0;
  virtual float GetTimeScale() = 0;

  virtual bool GetDebugLabel() = 0;
  virtual void SetDebugLabel( bool value ) = 0;

  virtual void OverloadPlayerModel(const char *modelname){};

  //============================================================================
  // Player Inventory
  //============================================================================

  virtual int           InventoryGetItemCount( int player ) = 0;
  virtual const char*   InventoryGetItemName( int player,int item ) = 0;
  virtual const char*   InventoryGetItemDescription( int player, int item ) = 0;
  virtual int           InventoryGetItemQuantity( int player, int item ) = 0;
  virtual const char*   InventoryGetItemIcon( int player,int item ) = 0;
  virtual int           InventoryGetItemType( int player, int item ) = 0;
  virtual int           InventoryGetItemValue( int player, int item ) = 0;
  virtual int           InventoryGetItemDamageBonus( int player, int item ) = 0;
  virtual int           InventoryGetItemDefenseBonus( int player, int item ) = 0;
  virtual int           InventoryGetGold( int player ) = 0;
  virtual bool          InventoryIsEquipped( int player, int item ) = 0;
  virtual bool          InventoryEquip( int player, int item ) = 0;
  virtual void          InventoryUnequip( int player, int item ) = 0;
  virtual bool          InventoryDrop( int player, int item ) = 0;

  //============================================================================
  // Scene options
  //============================================================================

  // Gets or sets a value that indicates whether the bloom post processing effect is enabled
  virtual bool GetBloom() = 0;
  virtual void SetBloom( bool value ) = 0;

  virtual bool GetMultiPass() = 0;
  virtual void SetMultiPass( bool value ) = 0;

  enum DrawMode
  {
    DrawModeNormal = 0,
    DrawModeWire   = 1,
    DrawModeSolid  = 2,
  };
  virtual DrawMode GetDrawMode() = 0;
  virtual void SetDrawMode( DrawMode value ) = 0;

  virtual void SetEnableSky( bool b ) = 0;
  virtual bool GetEnableSky() = 0;


  //============================================================================
  // Performance Stats
  //============================================================================

  virtual int GetFPS() = 0;
  virtual int GetBatches() = 0;
  virtual int GetVertexes() = 0;
  virtual int GetTriangles() = 0;
  virtual uint32 GetPixels() = 0;
  virtual int GetDegenerateTriangles() = 0;

  virtual int GetAverageFPS() = 0;
  virtual int GetAverageBatches() = 0;
  virtual int GetAverageVertexes() = 0;
  virtual int GetAverageTriangles() = 0;
  virtual int GetAverageDegenerateTriangles() = 0;

  //============================================================================
  // Cut Scene Control
  //============================================================================
  virtual void PlayCutScene(char8*) = 0;

  //============================================================================
  // Player Stats
  //============================================================================

  virtual int GetPlayerHealthMax() = 0;
  virtual int GetPlayerHealth() = 0;
  virtual int GetPlayerBlockMax() = 0;
  virtual int GetPlayerBlock() = 0;

  //============================================================================
  // UI Activate String
  //============================================================================

  virtual bool GetActivateString(char *buffer,int buffLen){return false;};

};

/**/

#endif // __TITANI_H__

