/********************************************************************

  Module: TitanT4File

  Author: Carlo Ordonez

  Description:

    Platform-specific file I/O for T4 data files

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TitanT4File.h"
#include "SyFile.h"
#include "SyAssert.h"
#include "SyNew.h"

//***********************************************************************
// Constructors
//***********************************************************************

TitanT4File::TitanT4File()
{
  mpFile = NULL;
}

TitanT4File::TitanT4File( const char* pFilename, const char* pMode )
{
  mpFile = SyNew SyFile();
  if( mpFile )
  {
    // assume only binary reads
    if( mpFile->Open( pFilename, SYFILE_RONLY | SYFILE_BINARY | SYFILE_ASYNC ) < 0 )
    {
      delete mpFile;
      mpFile = NULL;
    }
  }
}

//***********************************************************************
// Destructor
//***********************************************************************

TitanT4File::~TitanT4File()
{
  if( mpFile )
  {
    mpFile->Close();
    delete mpFile;
    mpFile = NULL;
  }
}

//***********************************************************************
// error
//  Returns -1 if there is no valid file open for this instance
//***********************************************************************

int TitanT4File::error()
{
  if( mpFile )
  {
    return( 0 );
  }
  else
  {
    return( -1 );
  }
}

//***********************************************************************
// isBusy
//***********************************************************************

bool TitanT4File::isBusy()
{
  return( mpFile && !mpFile->IsAsynchComplete() );
}

//***********************************************************************
// load
//  Asynchronous read
//***********************************************************************

int TitanT4File::load( long Offset, void* pBuffer, size_t NumBytes )
{
  if( mpFile )
  {
    return( mpFile->ReadAsynch( Offset, pBuffer, (int)NumBytes ) );
  }
  else
  {
    return( -1 );
  }
}

//***********************************************************************
// open
//  Creates a new TitanT4File instance
//***********************************************************************

T4File* TitanT4File::open( const char* pFilename, const char* pMode )
{
  TitanT4File* pT4File = new TitanT4File( pFilename, pMode );
  if( pT4File == NULL )
  {
    // out of memory?
    return( NULL );
  }
  else if( pT4File->error() == 0 )
  {
    // good, no errors
    return( pT4File );
  }
  else
  {
    // couldn't open file
    delete pT4File;
    return( NULL );
  }
}

//***********************************************************************
// read
//***********************************************************************

size_t TitanT4File::read( void* pBuffer, size_t Size, size_t NumObjects )
{
  if( mpFile && mpFile->Read( pBuffer, (int)(Size * NumObjects) ) == 0 )
  {
    return( NumObjects );
  }
  else
  {
    return( 0 );
  }
}

//***********************************************************************
// seek
//***********************************************************************

int TitanT4File::seek( long Offset, int Origin )
{
  if( mpFile )
  {
    // convert from CRT origin to SyFile SeekOrigin
    SyFile::SeekOrigin seekPos = SyFile::Set;
    switch( Origin )
    {
    case SEEK_CUR:
      seekPos = SyFile::Cur;
    	break;
    case SEEK_END:
      seekPos = SyFile::End;
      break;
    }

    return( mpFile->Seek( Offset, seekPos ) );
  }
  else
  {
    // no file
    return( -1 );
  }
}

//***********************************************************************
// tell
//***********************************************************************

long TitanT4File::tell()
{
  if( mpFile )
  {
    return( (long)mpFile->Tell() );
  }
  else
  {
    return( -1 );
  }
}
