/********************************************************************

  Module: TiEquipWindow

  Author: Carlo Ordonez

  Description:

    Equippable item window

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiEquipWindow.h"
#include "TiItemDescription.h"
#include "TiItemWindow.h"
#include "TitanI.h"

IMPLEMENT_CLASS( TiEquipWindow, TiWindow )

BEGIN_EVENT_TABLE( TiEquipWindow, TiWindow )
EVT_KEY_DOWN( TiEquipWindow::OnKeyDown )
EVT_PAINT( TiEquipWindow::OnPaint )
EVT_TIWINDOW_CLOSE( ID_TITAN_ITEM_LIST, TiEquipWindow::ItemList_Close )
EVT_TIITEM_ACTIVATE( TiEquipWindow::ItemList_Activate )
EVT_TIITEM_DROP( TiEquipWindow::ItemList_Drop )
END_EVENT_TABLE()

//***********************************************************************
// TiEquipWindow::Slot
//  internal class
//***********************************************************************

TiEquipWindow::Slot::Slot( const wxString& Name, const wxString& Image, EquipSlot EquipSlot )
{
  mName = Name;
  mImage = Image;
  mEquipSlot = EquipSlot;
}

int TiEquipWindow::Slot::GetHeight() const
{
  wxBitmap bmp( mImage );
  return( bmp.GetHeight() );
}

int TiEquipWindow::Slot::GetWidth() const
{
  wxBitmap bmp( mImage );
  return( bmp.GetWidth() );
}

void TiEquipWindow::Slot::Draw( wxDC* pDC, wxPoint Location, bool Selected )
{
  // draw image
  wxBitmap bmp( mImage );
  pDC->DrawBitmap( bmp, Location );

  // TODO: proper selection graphic
  if( Selected )
  {
    wxBitmap sel( wxT("test/selection.tga") );
    pDC->DrawBitmap( sel, Location );
  }
  ///////
}

//***********************************************************************
// Constructor
//***********************************************************************

TiEquipWindow::TiEquipWindow( wxWindow* pParent, TitanI* pTitan, int PlayerId, const wxString& Name, int WindowId )
: TiWindow( pParent, pTitan, PlayerId, Name, WindowId )
{
  mSlots.push_back( Slot(_("Melee"),      _("test/melee.tga"),      kMelee ) );
  mSlots.push_back( Slot(_("Ranged"),     _("test/ranged.tga"),     kRanged ) );
  mSlots.push_back( Slot(_("Head"),       _("test/head.tga"),       kHead ) );
  mSlots.push_back( Slot(_("Body"),       _("test/body.tga"),       kBody ) );
  mSlots.push_back( Slot(_("Feet"),       _("test/feet.tga"),       kFeet ) );
  mSlots.push_back( Slot(_("Accessory"),  _("test/accessory.tga"),  kAccessory ) );

  mSelection = 0;

  mpDescription = new TiItemDescription( this, pTitan, PlayerId );

  mpItemWindow = new TiItemWindow( this, pTitan, PlayerId );
  mpItemWindow->Show( false );
}

//***********************************************************************
// Destructor
//***********************************************************************

TiEquipWindow::~TiEquipWindow()
{
  mSlots.clear();
  mpDescription->Destroy();
  mpItemWindow->Destroy();
}

//***********************************************************************
// Next
//***********************************************************************

void TiEquipWindow::Next()
{
  if( mSelection < kMax-1 )
  {
    mSelection++;
  }
}

//***********************************************************************
// Prev
//***********************************************************************

void TiEquipWindow::Prev()
{
  if( mSelection > 0 )
  {
    mSelection--;
  }
}

//***********************************************************************
// Select
//***********************************************************************

void TiEquipWindow::Select()
{
  // open item list
  mpItemWindow->SetItemType( mSelection );
  mpItemWindow->Show( true );
  mpItemWindow->SetFocus();
}

//***********************************************************************
// CloseItemList
//***********************************************************************

void TiEquipWindow::CloseItemList()
{
  mpItemWindow->Show( false );
  this->SetFocus();
}

//***********************************************************************
// GetEquippedItem
//***********************************************************************

int TiEquipWindow::GetEquippedItem( EquipSlot Slot )
{
  int count = mpTitan->InventoryGetItemCount( mPlayerId );
  for( int i = 0; i < count; i++ )
  {
    if( mpTitan->InventoryIsEquipped( mPlayerId, i ) &&
        mpTitan->InventoryGetItemType( mPlayerId, i ) == Slot )
    {
      // found
      return( i );
    }
  }

  // none
  return( -1 );
}

//***********************************************************************
// OnKeyDown
//***********************************************************************

void TiEquipWindow::OnKeyDown( wxKeyEvent& event )
{
  switch( event.GetKeyCode() )
  {
  case WXK_UP:
    Prev();
  	return;

  case WXK_DOWN:
    Next();
    return;

  case WXK_RETURN:
    Select();
    return;
  }

  // default
  GetParent()->ProcessEvent( event );
}

//***********************************************************************
// OnPaint
//***********************************************************************

void TiEquipWindow::OnPaint( wxPaintEvent& event )
{
  TiWindow::OnPaint( event );

  wxPaintDC dc( this );

  PrepareDC( dc );
  dc.BeginDrawing();
  {
    const int BORDER = 5;
    wxPoint pt( BORDER, BORDER );
    wxPoint descLoc = pt;

    // draw each equip slot category
    list<Slot>::iterator it;
    for( it = mSlots.begin(); it != mSlots.end(); it++ )
    {
      bool selected = mSelection == it->mEquipSlot;
      it->Draw( &dc, pt, selected );
      if( selected )
      {
        descLoc = wxPoint( pt.x + it->GetWidth() + BORDER, pt.y );
        mpDescription->SetTitle( wxString::Format( "Current Item - %s", it->mName.ToAscii() ) );
        mpDescription->SetItemHandle( GetEquippedItem( (EquipSlot)mSelection ) );
      }

      pt.y += ( BORDER + it->GetHeight() );
    }

    // item description
    mpDescription->SetPosition( descLoc );

    // item list
    wxPoint listLoc = descLoc;
    listLoc.x += BORDER + mpDescription->GetSize().GetWidth();
    mpItemWindow->SetPosition( listLoc );
  }
  dc.EndDrawing();
}

//***********************************************************************
// ItemList_Close
//***********************************************************************

void TiEquipWindow::ItemList_Close( TiWindowEvent& event )
{
  CloseItemList();
}

//***********************************************************************
// ItemList_Activate
//***********************************************************************

void TiEquipWindow::ItemList_Activate( TiItemEvent& event )
{
  int itemHandle = event.GetItemHandle();
  if( mpTitan->InventoryIsEquipped( mPlayerId, itemHandle ) )
  {
    // un-equip item
    mpTitan->InventoryUnequip( mPlayerId, itemHandle );
  }
  else
  {
    // equip item
    mpTitan->InventoryEquip( mPlayerId, itemHandle );
  }
}

//***********************************************************************
// ItemList_Drop
//***********************************************************************

void TiEquipWindow::ItemList_Drop( TiItemEvent& event )
{
  int itemHandle = event.GetItemHandle();
  if( mpTitan->InventoryDrop( mPlayerId, itemHandle ) )
  {
    mpItemWindow->OnDrop( event );
  }
}
