/******************************************************************

  Module: Titan.h

  Author: 

  Description:

  Titan class definition

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*******************************************************************/

#ifndef __TITAN_H__
#define	__TITAN_H__

/*
**
**	 Forward Declarations
**
*/

class TitanPeeringNetwork;
class TitanInputHandler;
class TitanFXScriptDriver;
class cGameObjectRegistry;
class cGameObject;
class cCameraController;
class cDatabaseSys;
class cRuleSys;
class cAnimControllerSys;
class cRuleContainer;
class cInventoryRuleContainer;
class cTuningSys;
class cScriptSys;
class cStatsItemMaster;
class cItem;
class cAreaEffectSys;
class cDebrisSys;

/**/

/*
**
**	 Includes
**
*/

#include "TitanI.h"
                  

/**/
static const int MAX_CONTROLLERS = 4;
/*
**
**	 Type Definitions
**
*/



class Titan : public TitanI
{
public:
  // Initializes an instance of the Titan class.
  Titan( TitanUII* pTitanUI );
  
  // Destroys an instance of the Titan class.
  virtual ~Titan(void);

  // Allocates resources required by the Titan object.
  virtual bool Init( uint64 time );

  // Releases resources acquired by the titan object.
  virtual void Close();

  // Updates the game state to the specified time.
  virtual void Update( uint64 time );


  // Renders the scene.
  // Scene::BeginScene must be called before this method is invoked.
  // Scene::EndScene must be called after this method is invoked.
  virtual void Render();

  virtual TitanInputHandlerI* GetInputHandler( int controllerId );

  // use the specified XML file to construct all the initial game objects
  virtual void LoadLevel(const char *xmlFilename);

  // use the specified saved-game file to construct all the initial game object
  virtual void LoadSavedGame(const char *saveGameFilename);

  // restarts the current level and resets the player
  virtual void Restart();

  // tells it to join the specified existing network game
  // after network game is joined, you don't need to do anything else as all the initial state
  // will come from the peers in the game you are joining
  virtual void NetworkGameJoin(/*gift*/ PeeringNetwork::PeeringServerApi *api, int gameId);

  // tell it to create a new network game (and join it)
  // after the network game is created, you must call LoadXml or LoadSavedGame to populated the newly created game
  virtual void NetworkGameCreate(/*gift*/ PeeringNetwork::PeeringServerApi *api, const char *gameName);

  // this is a temporary hack function to startup a network game until the front-end is completed
  // once the front-end is completed, it will create the PeeringServerApi, do stuff with it, and then
  // pass it off via the NetworkGameJoin or NetworkGameCreate calls
  // This function will join the game with the specified name if it exists, otherwise it will create a game
  // with the specified name using the xmlFilename to setup the initial state of the world
  virtual void HackStartupNetwork(const char *playerName, const char *gameName, const char *xmlFilename, const char *recordFilename, const char *playbackFilename);


  // enables source level debuggin for Pawn scripting system
  virtual void ScriptEnableDebugging();

   // process cheat key
  virtual void CheatKey(int key);

  //============================================================================
  // Design Settings
  //============================================================================

  // Sets whether to log combat events and calculations to a file.
  virtual void SetLogMode(bool on);
  virtual bool GetLogMode();

  // Sets whether to log combat events and calculations to a file.
  virtual void SetCheckArt(bool on);
  virtual bool GetCheckArt();

  // Sets whether player is invulnerable or not
  virtual void SetGodMode(bool on);
  virtual bool GetGodMode();

  // Sets whether to log combat events and calculations to a file.
  virtual void SetInvisibleMode(bool on);
  virtual bool GetInvisibleMode();

  // set whether or not game is paused...
  virtual void PauseGame();
  virtual void UnpauseGame();
  virtual bool IsPaused();

  virtual void SetTimeScale(float scale);
  virtual float GetTimeScale();

  // Gets or sets a value that indicates whether floating debug information is enabled
  virtual bool GetDebugLabel(){return mDebugLabelEnabled;}
  virtual void SetDebugLabel( bool value ){mDebugLabelEnabled = value;};

  virtual void OverloadPlayerModel(const char *modelname);

  //============================================================================
  // Player Inventory
  //============================================================================

  virtual int           InventoryGetItemCount( int player );
  virtual const char*   InventoryGetItemName( int player, int item );
  virtual const char*   InventoryGetItemDescription( int player, int item );
  virtual int           InventoryGetItemQuantity( int player, int item );
  virtual const char*   InventoryGetItemIcon( int player, int item );
  virtual int           InventoryGetItemType( int player, int item );
  virtual int           InventoryGetItemValue( int player, int item );
  virtual int           InventoryGetItemDamageBonus( int player, int item );
  virtual int           InventoryGetItemDefenseBonus( int player, int item );
  virtual int           InventoryGetGold( int player );
  virtual bool          InventoryIsEquipped( int player, int item );
  virtual bool          InventoryEquip( int player, int item );
  virtual void          InventoryUnequip( int player, int item );
  virtual bool          InventoryDrop( int player, int item );

  //============================================================================
  // Scene options
  //============================================================================

  // Gets or sets a value that indicates whether the bloom post processing effect is enabled
  virtual bool GetBloom();
  virtual void SetBloom( bool value );

  virtual bool GetMultiPass();
  virtual void SetMultiPass( bool value );

  virtual DrawMode GetDrawMode();
  virtual void SetDrawMode( DrawMode value );

  virtual void SetEnableSky( bool b );
  virtual bool GetEnableSky();

  //============================================================================
  // Performance Stats
  //============================================================================

  virtual int GetFPS();
  virtual int GetBatches();
  virtual int GetVertexes();
  virtual int GetTriangles();
  virtual uint32 GetPixels();
  virtual int GetDegenerateTriangles();

  virtual int GetAverageFPS();
  virtual int GetAverageBatches();
  virtual int GetAverageVertexes();
  virtual int GetAverageTriangles();
  virtual int GetAverageDegenerateTriangles();

  //============================================================================
  // Cut Scene Methods
  //============================================================================

  virtual void PlayCutScene(char8* cutSceneName);
  virtual bool isCutScenePlaying();

  //============================================================================
  // Player Stats
  //============================================================================

  virtual int GetPlayerHealthMax();
  virtual int GetPlayerHealth();
  virtual int GetPlayerBlockMax();
  virtual int GetPlayerBlock();

  //============================================================================
  // UI Activate String
  //============================================================================

  virtual bool GetActivateString(char *buffer,int buffLen);

  //============================================================================
  // Implementation From Here On
  //============================================================================

  // Game Interface : Not exposed to UI or outside of Titan project

  SyScene*              GetScene(){return mpTitanUI->GetScene();} // convenience function gets scene from TitanUI
  cGameObjectRegistry * GetRegistry(){return mpRegistry;}
  cDatabaseSys *        GetDatabaseSys(){return mpDatabaseSys;}
  cRuleSys *            GetRuleSys(){return mpRuleSys;}
  TitanUII  *           GetTitanUI(){return mpTitanUI;}
  cCameraController *   GetCameraController(){return mpCameraController;}
  cAnimControllerSys *  GetAnimControllerSys(){return mpAnimControllerSys;}
  cRuleContainer *      GetGlobalRules(){return mpGlobalRules;}
  cInventoryRuleContainer *     GetInventoryRules(){return mpInventoryRules;}
  TitanPeeringNetwork * GetPeeringNetwork(){return mpTitanPeeringNetwork;}
  cScriptSys *          GetScriptSys(){return mpScriptSys;};
  TitanFXScriptDriver * GetFXScriptDriver(){return mpTitanFXScriptDriver;};
  cAreaEffectSys  *     GetAreaEffectSys(){return mpAreaEffectSys;};
  cDebrisSys *          GetDebrisSys(){return mpDebrisSys;};



  void                  SaveGame(); // todo: Expose to ui?  Come up with arguments (slotnumber?)
  void                  LoadGame(); // todo: Expose to ui?  Come up with arguments (slotnumber?)
  
  int                   Random(int min,int max);
  float                 RandomFloat(); // between 0 and 1
  uint64                GetTime() { return(mCurTime); }
  // stops update processing; used when game has a modal window open
  void                  SetUnfocused(bool unfocus){mbUnfocused = unfocus;};

                                  
protected:

  void                  PostLoad();
  void                  LoadLevelScript(const char *filename);

  const cStatsItemMaster* GetItemMaster( int player, int item );   
  int                   SelectItemEquipSlot( cItem* pItem );

  void                  IncreaseTimeScale();
  void                  DecreaseTimeScale();

private:
  // Keep header dependencies under control, please don't embed structures
  // in this object, just pointers and ubiquitous types like int, float, SyString, etc.

  TitanUII*             mpTitanUI;

  TitanInputHandler*    mpInputHandler;

  cGameObjectRegistry*  mpRegistry;

  cCameraController*    mpCameraController;

  cDatabaseSys*         mpDatabaseSys; 

  cRuleSys*             mpRuleSys;

  cAnimControllerSys*   mpAnimControllerSys;
  
  cRuleContainer *      mpGlobalRules;
  cInventoryRuleContainer * mpInventoryRules;

  TitanPeeringNetwork * mpTitanPeeringNetwork;

  cScriptSys *          mpScriptSys;

  cAreaEffectSys *      mpAreaEffectSys;

  cDebrisSys *          mpDebrisSys;

  bool                  mbPaused; // true if game is currently paused
  uint64                mCurTime; // current UI system time
  uint64                mGameTime; // millisecs not spent in pause since Init
  float                 mTimeScale; // 1.0f = time progresses at normal rate

  bool                  mBloomEnabled;
  int                   mBloomEffectHandle;
  bool                  mMultiPassEnabled;
  DrawMode              mDrawMode;
  TitanFXScriptDriver*  mpTitanFXScriptDriver;
  char *                mCurFilename;
  bool                  mDebugLabelEnabled;
  bool                  mbUnfocused;
  bool                  mbCheckArt;
};


#endif // __TITAN_H__








