/******************************************************************
  
  Module:  gameobject.h
  
  Author: Sean Craig
  
  Description:  Class declarations for game object, the base object
   class in the Titan game. 

  
  Copyright 2005 Sony Online Entertainment.  All rights reserved.
  
*******************************************************************/

#ifndef GAMEOBJECT_H
#define GAMEOBJECT_H

//-------------------------------------------------------- Includes
#include "SyPropObject.h"
#include "nameid.h"
#include "script_func.h"

//-------------------------------------------------------- Typedefs

typedef tGameID tGameObjectID;

//---------------------------------------------- Class Declarations
class cGameObjectRegistry;
class Titan;
class cIntel;
class cStats;
class cInventory;
class cGraphic;
class cPhysics;
class cScript;

static const int GAMEOBJECT_CLASSID  =  0xFFFF0001;

class cPosition : public SyPropObject
{
public:
  /* Property Class Support */
  static int           InitPropClass(){return 0;};   // TODO
  static SyPropObject* Creator(){ return 0; };  // TODO

  // public interface
 
  void            SetLocation(const SyVect3 &loc){mLocation = loc;};
  void            SetHPR(const SyVect3 &hpr){mHPR = hpr;};
  void            SetHeading(const float heading) {mHPR(0) = heading;};
  void            SetPitch(const float pitch) {mHPR(1) = pitch;};
  void            SetRoll(const float roll) {mHPR(2) = roll;};

  const SyVect3 & GetLocation() {return mLocation;};
  const SyVect3 & GetHPR() {return mHPR;};
  float           GetHeading(){return mHPR(0);};
  float           GetPitch(){return mHPR(1);};
  float           GetRoll(){return mHPR(2);};


protected:

  SyVect3       mLocation;
  SyVect3       mHPR;
};



class cGameObject : public SyPropObject
{
public:

  typedef enum
  {
    OBJ_PLAYER,
    OBJ_NPC,
    OBJ_ITEM, // in the world
    OBJ_MARKER,
    OBJ_PROJECTILE,
    OBJ_PROP,
    NUM_OBJ_TYPES
  } tObjectType;


                    cGameObject();
                   ~cGameObject();

  /* Property Class Support */
  static int           InitPropClass();
  static SyPropObject* Creator();  // TODO

  // public interface

  tGameObjectID     Create(cGameObjectRegistry *owner,tGameObjectID id,tObjectType newtype); // creates and registers subobjects
  void              SetName(const char *newname){mName.Init(newname);};
  tObjectType       GetType(){return mType;};

  void              Init(); // call Create, set up variables and subobjects, then call Init
  void              Exit(); // opposite of init

  void              Reset(); // for example, when respawning
  void              Update(float time); // once/frame
  bool              CheckForDelete();
  void              Prerender(); // after update, before render

  void              Activate(cGameObject *activater);
  void              Open();

  cIntel     *      GetIntel(){return mIntel;};
  cStats     *      GetStats(){return mStats;};
  cInventory *      GetInventory(){return mInventory;};
  cGraphic   *      GetGraphic(){return mGraphic;};
  cPhysics   *      GetPhysics(){return mPhysics;};
  cScript    *      GetScript(){return mScript;};

  const SyVect3 &   GetLocation(){return mPosition.GetLocation();};
  const SyVect3 &   GetHPR(){return mPosition.GetHPR();};
  float             GetHeading()  {return mPosition.GetHeading();};
  float             GetPitch(){return mPosition.GetPitch();};
  float             GetRoll() {return mPosition.GetRoll();};

  void              SetLocation(const SyVect3 &loc){mPosition.SetLocation(loc);};
  bool              SetSafeLocation(const SyVect3 &loc); // finds a safe location near this spot
  void              SetHPR(const SyVect3 &hpr){mPosition.SetHPR(hpr);};
  void              SetHeading(float radians)  {return mPosition.SetHeading(radians);};
  void              SetPitch(float radians){return mPosition.SetPitch(radians);};
  void              SetRoll(float radians) {return mPosition.SetRoll(radians);};
  

  void              SetID(tGameObjectID id){mID = id;}; // be careful don't leave a dangling reference in the registry.
  tGameObjectID     GetID(){return mID;};
  tGameID           GetNameID();
  const char *      GetName(){return mName.AsChar();};
  const char *      GetModel();
  tGameID           GetAnimSet();

  cGameObjectRegistry *GetRegistry(){return mOwner;};
  Titan *           GetTitan();

  void AddCondition(const char* name, int numParams, cell* pParams);
  void RemoveCondition(const char* name);

  float             GetDistance(cGameObject *other);
  float             GetDistance(const SyVect3 other_loc);
  float             GetHeadingTowards(cGameObject *other);
  float             GetHeadingTowards(const SyVect3 &other_loc);
  float             GetPitchTowards(cGameObject *other);

  void              FixUp(cGameObjectRegistry *owner);
  int               Reset( SyObjFile& File ); // resets to state in the file 
  int               GetState( SyObjFile& File); 

  bool              IsRemote(){return mRemote;};
  bool              IsLocal(){return !IsRemote();};
  void              SetRemote(bool remote){mRemote = remote;};

  void              NetworkReceiveBroadcast(const char *packet,int size); // the ownint peer of this object has broadcast an update
  void              NetworkReceiveMessage(const char *packet,int size); // the ownint peer of this object has broadcast an update

  // property Ids for use in property editor, see SyPropObject::AddProperty() for numbering convention
  static const SyPropID PropId_ID           = 0x0000;
  static const SyPropID PropId_Type         = 0x0001;
  static const SyPropID PropId_Intel        = 0x0002;
  static const SyPropID PropId_Stats        = 0x0003;
  static const SyPropID PropId_Inventory    = 0x0004;
  static const SyPropID PropId_Graphic      = 0x0005;
  static const SyPropID PropId_Physics      = 0x0006;
  static const SyPropID PropId_Name         = 0x0007;
  static const SyPropID PropId_HPR          = 0x0008;
  static const SyPropID PropId_Location     = 0x0009;
  static const SyPropID PropId_Script       = 0x000a;

protected:
  void              NetworkUpdate(float time); 
  void              UpdateScope(); // check to see if an object 

  void              OnWake();
  void              OnSleep();

  tGameObjectID     mID;  // all game objects must have a unique one of these
  SyString          mName;  // some game objects are named as well
  tObjectType       mType;       
  
  bool              mRemote; // true if this object is controlled by another peer in a network game
  bool              mbInScope; // there is a "simulation bubble" around the camera.  Objects outside of this bubble are 'out of scope' and not simulated.
  float             mPositionUpdateTime;

  cGameObjectRegistry       * mOwner;

  cPosition                   mPosition;    
  cIntel                    * mIntel;  // controller input, AI, or network event handler
  cStats                    * mStats;  // game statistics, health, etc.
  cInventory                * mInventory; // item objects contained by this object
  cGraphic                  * mGraphic; // visual representation of object
  cPhysics                  * mPhysics; // controls motion of object
  cScript                   * mScript;  // various event handlers
};

//------------------------------------------- Function Declarations

void RegPropClasses_GameObj();
//--------------------------------------------------------- Globals
//------------------------------------------------ Inline Functions

#endif
