/******************************************************************
  
  Module:  Registry.h
  
  Author: Sean Craig
  
  Description:  Registry for game objects

  
  Copyright 2005 Sony Online Entertainment.  All rights reserved.
  
*******************************************************************/

#ifndef REGISTRY_H
#define REGISTRY_H

//-------------------------------------------------------- Includes

#include "gameobj.h"
#include "SyMap.h"
#include "SyString.h"
#include "SySceneTypes.h"

//-------------------------------------------------------- Typedefs
//---------------------------------------------- Class Declarations
class Titan;
class SyLoader;
class cAIBlackboard;

static const int GAMEOBJECTREGISTRY_CLASSID  =  0xFFFFA001;
static const int LEVELOBJECT_CLASSID         =  0xFFFFA002;
class cLevelObject : public SyPropObject
{
public: 
  cLevelObject();
  ~cLevelObject();

  /* Property Class Support */
  static int           InitPropClass();
  static SyPropObject* Creator();  

  const char *      GetCameraName(){return mCameraName.AsChar();};
  void              SetCameraName(const char *name);

  const char *      GetModelName(){return mModelName.AsChar();};
  void              SetModelName(const char *name);
  void              Init(Titan *titan);

  Titan*            GetTitan() {return mpTitan;}
  tGameID           GetID() {return mID;}

  void              Update();

  void              NetworkReceiveBroadcast(const char *state, int maxlen);
  void              NetworkReceiveMessage(const char *state, int maxlen);

  void              SetRemote(bool bRemote);

  static const int LEVEL_OBJECT_GAME_ID = 0x1;
protected:
  tGameID           mID; // used to synch up across network..
  SyString          mModelName;  // the level model, geometry + textures for the world
  SyString          mCameraName;
  Titan *           mpTitan;
  SyLoader*         mpLoader;
  cAIBlackboard*    mpAIBlackboard;
};

class cGameObjectRegistry : public SyPropObject
{
public:
                    cGameObjectRegistry();
                    ~cGameObjectRegistry();

  void              Clear();

                    /* Property Class Support */
  static int           InitPropClass();
  static SyPropObject* Creator();  
  
  void              Init(Titan *parent){mpTitan = parent;}; 
  void              LoadLevelXML(const char *xml_filename);  // call before InitGameObjects
  void              LoadLevelBinary(const char *filename);   // call before InitGameObjects
                                     
  void              SaveGame();
  void              LoadGame();

  void              InitObject(tGameObjectID id,bool ownerLifetime);
  void              Update(float time);
  void              Prerender();
  
  tGameObjectID     Create(cGameObject::tObjectType newtype); // creates and registers objects
  int               Delete(tGameObjectID);
  cGameObject *     Fetch(tGameObjectID id);
  cGameObject *     Fetch(const char *name);

  cGameObject *     FetchByActorHandle(SyActorHandle handle); 

  tGameObjectID     NetworkCreate(cGameObject::tObjectType newtype,tGameObjectID id,const char *state,int statelen); // creates and registers objects
  void              NetworkSetState(tGameObjectID id,const char *state,int statelen); // Resets an object to a particular state
  unsigned int      NetworkGetState(tGameObjectID id,char *state, int maxlen);
  void              NetworkDelete(tGameObjectID id);
  unsigned int      NetworkReceiveBroadcast(tGameObjectID id,const char *state, int maxlen);
  unsigned int      NetworkReceiveMessage(tGameObjectID id,const char *state, int maxlen);
  unsigned int      NetworkTakeOwnership(tGameObjectID id);
  void              NetworkRequestOwnership(tGameObjectID id, int requestingPeerId);


  // for iterating over all world objects
  int               Begin(){return mObjects.Begin();};
  int               End(){return mObjects.End();};
  int               Next(int index){return mObjects.Next(index);};
  cGameObject *     operator()(int Node) {return mObjects(Node);};

  Titan *           GetTitan(){return mpTitan;};
  cLevelObject*     GetLevelObject(){return mpLevelObject;};


  bool              Name(cGameObject *obj,const char *name);  // assign a name to an object
 protected:
  void              CheckForDelete(); // checks all objects to see if they should be deleted...
  bool              Add(cGameObject *obj);


  Titan *           mpTitan;

  SyMap<tGameObjectID,cGameObject *>  mObjects;
  SyMap<tGameID,cGameObject *>        mNames;  // for looking up objects by name

  cLevelObject *    mpLevelObject;

};

//------------------------------------------- Function Declarations
//--------------------------------------------------------- Globals
void RegPropClasses_Registry();


//------------------------------------------------ Inline Functions

#endif
