/********************************************************************

  Module: TiItemWindow

  Author: Carlo Ordonez

  Description:

    List of items

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiItemWindow.h"
#include "TiItemDescription.h"
#include "TiListView.h"
#include "TiWindowEvent.h"
#include "TiItemEvent.h"
#include "TitanI.h"

IMPLEMENT_CLASS( TiItemWindow, TiWindow )

BEGIN_EVENT_TABLE( TiItemWindow, TiWindow )
EVT_PAINT( TiItemWindow::OnPaint )
EVT_KEY_DOWN( TiItemWindow::OnKeyDown )
EVT_SET_FOCUS( TiItemWindow::OnSetFocus )
END_EVENT_TABLE()

//***********************************************************************
// Constructor
//***********************************************************************

TiItemWindow::TiItemWindow( wxWindow* pParent, TitanI* pTitan, int PlayerId, const wxString& Name, int WindowId )
: TiWindow( pParent, pTitan, PlayerId, Name, WindowId )
{
  wxBoxSizer* pSizer = new wxBoxSizer( wxVERTICAL );
  SetSizer( pSizer );

  int growFactor = 1;
  int border = 5;

  mpListView = new TiListView( this, ID_TITAN_ITEM_LIST_VIEW, wxLC_REPORT|wxLC_SINGLE_SEL|wxLC_NO_HEADER );
  mpListView->InsertColumn( 0, _("Items") );
  pSizer->Add( mpListView, growFactor, wxGROW|wxALL, border );

  mpDescription = new TiItemDescription( this, pTitan, PlayerId );
  mpDescription->SetTitle( _("Item") );
  pSizer->Add( mpDescription, growFactor, wxGROW|wxALL, border );

  pSizer->SetSizeHints( this );
  Layout();
}

//***********************************************************************
// Destructor
//***********************************************************************

TiItemWindow::~TiItemWindow()
{
  mpListView->Destroy();
  mpDescription->Destroy();
}

//***********************************************************************
// OnDrop
//***********************************************************************

void TiItemWindow::OnDrop( TiItemEvent& event )
{
  wxASSERT( mpListView->GetItemCount() > 0 );
  long sel = mpListView->GetFirstSelected();
  wxASSERT( sel >= 0 );
  wxASSERT( (int)mpListView->GetItemData(sel) == event.GetItemHandle() );
  mpListView->DeleteItem( sel );

  if( mpListView->GetItemCount() > 0 )
  {
    mpListView->Select( sel > 0 ? sel-1 : 0 );
  }
}

//***********************************************************************
// Select
//***********************************************************************

void TiItemWindow::Select()
{
  long sel = mpListView->GetFirstSelected();
  if( sel >= 0 )
  {
    int itemHandle = (int)mpListView->GetItemData( sel );
    TiItemEvent event( TIITEM_ACTIVATE, GetId() );
    event.SetItemHandle( itemHandle );
    ProcessEvent( event );
  }
}

//***********************************************************************
// Drop
//***********************************************************************

void TiItemWindow::Drop()
{
  long sel = mpListView->GetFirstSelected();
  if( sel >= 0 )
  {
    int itemHandle = (int)mpListView->GetItemData( sel );
    TiItemEvent event( TIITEM_DROP, GetId() );
    event.SetItemHandle( itemHandle );
    ProcessEvent( event );
  }
}

//***********************************************************************
// OnPaint
//***********************************************************************

void TiItemWindow::OnPaint( wxPaintEvent& event )
{
  TiWindow::OnPaint( event );
  wxPaintDC dc( this );

  long sel = mpListView->GetFirstSelected();
  if( sel >= 0 )
  {
    mpDescription->SetItemHandle( (int)mpListView->GetItemData(sel) );
  }
  else
  {
    mpDescription->SetItemHandle( -1 );
  }

  GetSizer()->SetSizeHints( this );
  Layout();
}

//***********************************************************************
// OnKeyDown
//***********************************************************************

void TiItemWindow::OnKeyDown( wxKeyEvent& event )
{
  switch( event.GetKeyCode() )
  {
  case WXK_UP:
    mpListView->Prev();
    return;

  case WXK_DOWN:
    mpListView->Next();
    return;

  case WXK_RETURN:
    Select();
    return;

  case WXK_DELETE:
    Drop();
    return;

  case WXK_ESCAPE:
    {
      TiWindowEvent event( TIWINDOW_CLOSE, GetId() );
      ProcessEvent( event );
    }
    return;
  }
}

//***********************************************************************
// OnSetFocus
//***********************************************************************

void TiItemWindow::OnSetFocus( wxFocusEvent& event )
{
  mpListView->DeleteAllItems();

  int count = mpTitan->InventoryGetItemCount( mPlayerId );
  for( int i = 0; i < count; i++ )
  {
    if( mpTitan->InventoryGetItemType( mPlayerId, i ) == mType )
    {
      wxListItem item;
      item.SetId( mpListView->GetItemCount() );
      item.SetText( _(mpTitan->InventoryGetItemName(mPlayerId,i)) );
      item.SetData( i );
      mpListView->InsertItem( item );
    }
  }
  if( mpListView->GetItemCount() > 0 )
  {
    mpListView->Select( 0 );
  }
}

