#ifndef __TICAMERADIRECTOR_H__
#define __TICAMERADIRECTOR_H__
/******************************************************************

  Module: TiCameraDirector.h

  Author: Bijan Forutanpour

  Description:   Game camera control interface.

  1) Use LoadCameraFile () to load in a .cam file. A .cam file is
     generated using the camExport Maya file exporter. 

  2) Given a 3d point (player, usually), return camera info.


  NOTES: Camera impelementations (work in progress)
     a) stationary camera
     b) stationary panning camera, following player
     c) stationary panning camera, lookat player  (and follow at
        fixed distance)
     d) Player-relative cam: fixed offset from player, fixed angle
     e) keyframed cameras.

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*******************************************************************/
#include <SyTypes.h>
#include <SyBBox.h>
#include <SyCamera.h>
#include <SyCurve.h>
#include <SyObjFile.h>
#include <SyTime.h>
#include "TiCamera.h"
#include "TiCameraZone.h"


/**********************************************************
 * TiCameraDirector - Used both by application (game) as 
 * well as artist camera simulation tool. The difference
 * in usage between the two sides is:
 * 
 * 1) Art side runs simulation first, then does playback of 
 *    the results at a constant frame rate (e.g. in Maya). 
 *    The simulation needs to index animation on a per-frame 
 *    basis.
 * 
 * 2) The game side runs in milliseconds, not frames. 
 *    So at render time the milliseconds are calculated first,
 *    then converted to frames for use in animation evaluation.
 *
 * The default is playback mode, ie, milliseconds.
 **********************************************************/
class TiCameraDirector
{
 private:
    TiCameraZone*      mCamZones;
    int32              mNumCamZones;
    int32              mLastZone;
    int32              mCurrentZone;
    float32            mTimeStep; 
    int32              mUseDefaultCamera;
    bool               mFrameIndexing;
    SyTime             mCurrentTime;

    void               LoadCurve (SyObjFile * syFile, SyCurve *theCurve);
    float32            DistancePointLine (SyVect3 Point, SyVect3 P1, SyVect3 P2, float32* dist);
    float32            getPointOnCurveU (SyVect3 playerPosition, SyCurve& curveX, SyCurve& curveY, SyCurve& curveZ);

 public:
    TiCameraDirector ();
    ~TiCameraDirector ();
    
    void UpdateTime (float deltaTime); 
		// used by game as well as art simulation tool
    int32 LoadCameraFile (const char8* camFileName);
    int32  GetCamInfo (SyVect3 playerPosition, SyVect3* camPos, SyVect3* camDir, float32 *fov);

    
    // Used by art simulation tool only
    void  SetTimeStep (float timeStep); // control the speed of anim playback
    int32  GetCamInfoInternal (SyVect3 pos, SyVect3* camPos, SyVect3* camInterest, int32 *orientType, float32 *fov);
    void  XYZtoHPR (SyVect3 euler, SyVect3* hpr);
		void  SetFrameIndexing (bool doFrames);
};

#endif
