/******************************************************************

  Module: TitanI.h

  Author: 

  Description:

  Titan interface.

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*******************************************************************/

#ifndef __TITANI_H__
#define	__TITANI_H__

/*
**
**	 Includes
**
*/

#include "SyTypes.h"
#include "SyVect2.h"
#include "SyVect2I.h"
#include "SyVect3.h"
#include "SyResourceID.h"
#include "SyString.h"
#include "SyTime.h"

/**/

/*
**
**	 Forward Declarations
**
*/

class SyScene;
class SyPerformanceStats;
class SyParameterValidator;
class SyESFParse;
class TiCommandMessage;
class SyInputConsole;
class TitanI;
class TiCharacterItemInfo;
class SyRaster;
class SyDictionary;
class TitanFXScriptDriver;

namespace PeeringNetwork { class PeeringServerApi; }


/**/

/*
**
**	 Type Definitions
**
*/

//==============================================================================
// Titan->TitanUI callback interface
//==============================================================================

enum TitanButton
{
  ButtonAttackL,
  ButtonAttackS,
  ButtonAction,
  ButtonBlock,
  ButtonAttackRanged,
  ButtonMagic,
  ButtonDodge,
  ButtonCameraCenter,
  ButtonJump,
  ButtonPause,
  ButtonDPadUp,
  ButtonDPadDown,
  ButtonDPadLeft,
  ButtonDPadRight,
  ButtonJoyLeft,

  // 
  ButtonMax
};

class TitanControllerI
{
public:
  // Destroys an instance of the TitanControllerI interface.
  virtual ~TitanControllerI() {}

  // returns true if the controller is connected.
  virtual bool GetConnected() const = 0;

  // returns true if the specified button is depressed.
  virtual bool GetButtonState( TitanButton tb ) const = 0;

  // returns a value in the range [-10000,+10000] that indicates whether
  // the directional control is pushed completely forward (+10000),
  // int backward (-10000) or somewhere in between.
  virtual int GetForward() const = 0;

  // returns a value in the range [-10000,+10000] that indicates whether
  // the directional control is pushed completely to the right (+10000),
  // completely to the left (-10000) or somewhere in between.
  virtual int GetRight() const = 0;

  // returns a value in the range [-10000,+10000] that indicates whether
  // the right directional control is pushed completely forward (+10000),
  // int backward (-10000) or somewhere in between.
  virtual int GetRightStickForward() const = 0;

  // returns a value in the range [-10000,+10000] that indicates whether
  // the right directional control is pushed completely to the right (+10000),
  // completely to the left (-10000) or somewhere in between.
  virtual int GetRightStickRight() const = 0;


  // changes joystick value to float [-1.0f,+1.0f] after removing "dead zone" in the middle
  virtual float Normalize(int in) const = 0;

  static const int DIRECTION_MAX =  10000;

};

class TiUII
{
public:
  virtual int32     GetBitmapInfo( const char8* pFilename, SyVect2I& SurfaceOffset, SyVect2I& Size) = 0;
  virtual int32     GetBitmapInfo( const char8* pFilename, int32& SurfaceHandle, SyVect2I& SurfaceOffset, SyVect2I& Size )= 0;
};

class TitanOptionsI
{
public:
  TitanOptionsI() {}
  virtual ~TitanOptionsI() {}

  // tunable settings
  virtual float32   GetBrightness() const { return 1.0f; }            // 0 to 1
  virtual void      SetBrightness( float32 Brightness ) {}

  virtual float32   GetAmbientVolume() const { return 1.0f; }         // 0 to 1
  virtual void      SetAmbientVolume( float32 Volume ) {}

  virtual float32   GetSoundFxVolume() const { return 1.0f; }         // 0 to 1
  virtual void      SetSoundFxVolume( float32 Volume ) {}

  // on/off settings
  virtual bool      GetCameraSwivel() const { return false; }         // camera follows over the shoulder?
  virtual void      SetCameraSwivel( bool On ) {}

  virtual bool      GetReverseCameraInputs() const { return false; }
  virtual void      SetReverseCameraInputs( bool On ) {}

  virtual bool      GetVibration() const { return true; }
  virtual void      SetVibration( bool On ) {}

  virtual bool      GetStereo() const { return true; }   
  virtual void      SetStereo( bool On ) {}

  virtual bool      GetBlood() const { return true; }    
  virtual void      SetBlood( bool On ) {}

  virtual bool      GetSubtitles() const { return true; }
  virtual void      SetSubtitles( bool On ) {}

  virtual bool      GetDamageText() const { return true; }
  virtual void      SetDamageText( bool On ) {}

  virtual bool      GetChatText() const { return true; }  
  virtual void      SetChatText( bool On ) {}

  virtual bool      GetPlayerMarkers() const { return true; }
  virtual void      SetPlayerMarkers( bool On ) {}
};

class TitanUII
{
public:
  // Destroys an instance of the TitanControllerI interface.
  virtual ~TitanUII() {}

  // Get the scene titan can use for drawing
  virtual SyScene* GetScene() = 0;
  virtual SyPerformanceStats* GetPerformanceStats() = 0;
  virtual SyInputConsole* GetInputConsole() = 0;
  virtual int GetBloomEffectHandle() = 0;

  // Get a pointer to an interface that can be used to poll a controller state
  virtual TitanControllerI* GetController( int controllerId ) = 0;

  virtual uint64 GetTime() = 0;

  // opens up a pawn debug UI
  virtual int  PawnDebugUI(const char *src, int linenum,const char *variables) = 0;
  virtual void PawnDebugOutput(const char *output){};

  // process a command from the UI
  virtual void ProcessCommandMessage( TiCommandMessage* pMessage ) {}

  // get the scene file parser
  virtual SyESFParse& GetESFParser() const = 0;
  virtual void UpdatePhysics( float deltatime ) = 0;

  // whether or not performance stats are showing
  virtual bool GetDisplayStats() const { return false; }

  // whether or not to show game window UIs (health bars, maps)
  virtual bool GetShowUI() const { return true; };

  // gets the ui
  virtual TiUII*       GetUII()=0;
  virtual TitanI      *GetTitanI() {return NULL;}

  // game time settings
  virtual TitanOptionsI* GetOptions() = 0;

  // minimap
  virtual void SetMinimap(const SyString&  textureName,
                          const SyVect2&   worldPos1,
                          const SyVect2&   worldPos2,
                          const SyVect2&   mapPos1,
                          const SyVect2&   mapPos2,
                          SyRaster*        raster,
                          SyDictionary*    dictionary) {}

  virtual void  BeginLetterbox(bool bInstant = false){};
  virtual void  EndLetterbox(bool bInstant = false){};
  virtual bool  IsLetterboxMode() const {return false;};
  virtual void  FadeOut(float32 fTransition) {}
  virtual void  FadeIn(float32 fTransition) {}
  virtual void  FadeAndExit(float32 fDelayBeforeFading, float32 fFadeDuration) {}
  virtual bool  IsExiting() {return false;}

  virtual void  SetPaused( int i ) {}
  virtual void  SetHavokStabilizeAndExit( int i ) {}
  virtual void  SetHavokRagdoll( int i ) {}
};

//==============================================================================
// TitanUI->Titan interface
//==============================================================================

class TitanInputHandlerI
{
public:
  // handle controller button state changes
  virtual void OnButtonDown( TitanButton tb ) = 0;
  virtual void OnButtonUp( TitanButton tb ) = 0;
};

class TitanSave
{
  public:
    SyString mName;
    SyTime mTimestamp;
    int   mNumNetworkPlayers;
    int   mNumLocalPlayers;
};

class TitanI
{
public:

  enum KeyCode
  {
    KEYCODE_NONE,

    KEYCODE_F1,
    KEYCODE_F2,
    KEYCODE_F3,
    KEYCODE_F4,
    KEYCODE_F5,
    KEYCODE_F6,
    KEYCODE_F7,
    KEYCODE_F8,
    KEYCODE_F9,
    KEYCODE_F10,
    KEYCODE_F11,
    KEYCODE_F12,

    KEYCODE_ESCAPE,
    KEYCODE_PRINT,
    KEYCODE_PAUSE,
    KEYCODE_ACCENT,
    KEYCODE_MINUS,
    KEYCODE_EQUAL,
    KEYCODE_BACKSPACE,
    KEYCODE_TAB,
    KEYCODE_LEFT_BRACKET,
    KEYCODE_RIGHT_BRACKET,
    KEYCODE_BACKSLASH,
    KEYCODE_SEMICOLON,
    KEYCODE_QUOTE,
    KEYCODE_ENTER,
    KEYCODE_COMMA,
    KEYCODE_PERIOD,
    KEYCODE_SLASH,
    KEYCODE_SPACE,
    KEYCODE_INSERT,
    KEYCODE_HOME,
    KEYCODE_PAGE_UP,
    KEYCODE_DELETE,
    KEYCODE_END,
    KEYCODE_PAGE_DOWN,
    KEYCODE_UP,
    KEYCODE_LEFT,
    KEYCODE_DOWN,
    KEYCODE_RIGHT,

    KEYCODE_0,
    KEYCODE_1,
    KEYCODE_2,
    KEYCODE_3,
    KEYCODE_4,
    KEYCODE_5,
    KEYCODE_6,
    KEYCODE_7,
    KEYCODE_8,
    KEYCODE_9,

    KEYCODE_A,
    KEYCODE_B,
    KEYCODE_C,
    KEYCODE_D,
    KEYCODE_E,
    KEYCODE_F,
    KEYCODE_G,
    KEYCODE_H,
    KEYCODE_I,
    KEYCODE_J,
    KEYCODE_K,
    KEYCODE_L,
    KEYCODE_M,
    KEYCODE_N,
    KEYCODE_O,
    KEYCODE_P,
    KEYCODE_Q,
    KEYCODE_R,
    KEYCODE_S,
    KEYCODE_T,
    KEYCODE_U,
    KEYCODE_V,
    KEYCODE_W,
    KEYCODE_X,
    KEYCODE_Y,
    KEYCODE_Z,

    KEYCODE_KEYPAD_0,
    KEYCODE_KEYPAD_1,
    KEYCODE_KEYPAD_2,
    KEYCODE_KEYPAD_3,
    KEYCODE_KEYPAD_4,
    KEYCODE_KEYPAD_5,
    KEYCODE_KEYPAD_6,
    KEYCODE_KEYPAD_7,
    KEYCODE_KEYPAD_8,
    KEYCODE_KEYPAD_9,
    KEYCODE_KEYPAD_PERIOD,
    KEYCODE_KEYPAD_DIVIDE,
    KEYCODE_KEYPAD_MULTIPLY,
    KEYCODE_KEYPAD_SUBTRACT,
    KEYCODE_KEYPAD_ADD,
    KEYCODE_KEYPAD_ENTER,

    KEYCODE_NUM_LOCK,
    KEYCODE_CAPS_LOCK,
    KEYCODE_SCROLL_LOCK,
    KEYCODE_SHIFT,
    KEYCODE_CTRL,
    KEYCODE_ALT,
    KEYCODE_WIN,
    KEYCODE_APPLICATION,
    KEYCODE_KANA,
    KEYCODE_HENKAN,
    KEYCODE_MUHENKAN,

    KEYCODE_COUNT       // count of how many keycodes there are (keep this entry last)
  };

  // Destroys an instance of the TitanI interface.
  virtual ~TitanI(void) {}

  // Allocates and constructs an instance of the TitanI interface.
  static TitanI* Create( TitanUII* pTitanUI );

  // Allocates resources required by the Titan object.
  virtual bool Init( uint64 time, const SyParameterValidator& parameterValidator ) = 0;

  // Releases resources acquired by the titan object.
  virtual void Close() = 0;

  // Updates the game state to the specified time.
  virtual void AdvanceTime( uint64 time ) {};
  virtual void Update( uint64 time ) = 0;

  // Updates the game state just for props to the specified time.
  virtual void UpdateJustProps( uint64 time ) = 0;

  // Renders the scene.
  // Scene::BeginScene must be called before this method is invoked.
  // Scene::EndScene must be called after this method is invoked.
  virtual void Render() = 0;

  // render the HUD
  virtual void RenderHUD() = 0;

  virtual TitanInputHandlerI* GetInputHandler( int controllerId ) = 0;

  virtual TitanFXScriptDriver* GetFXScriptDriver() { return NULL; }

  // use the specified file to construct all the initial game objects
  virtual void LoadLevel(const char *Filename) = 0;

  // save the binary of the level after physical simulation calms down 
  virtual void SaveLevel(const char8* filePrefix) = 0;

  // get the name of the current level
  virtual const char* GetLevelName() const = 0;

  // nudge every prop
  virtual void NudgeProps() = 0;

  // restarts the current level and resets the player
  virtual void Restart() = 0;

  // clear game resources before the next level load
  virtual void Reset() = 0;

  // tells it to join the specified existing network game
  // after network game is joined, you don't need to do anything else as all the initial state
  // will come from the peers in the game you are joining
  virtual void NetworkGameJoin(/*gift*/ PeeringNetwork::PeeringServerApi *api, int gameId) = 0;

  // tell it to create a new network game (and join it)
  // after the network game is created, you must call LoadXml or LoadSavedGame to populated the newly created game
  virtual void NetworkGameCreate(/*gift*/ PeeringNetwork::PeeringServerApi *api, const char *gameName) = 0;

  // this is a temporary hack function to startup a network game until the front-end is completed
  // once the front-end is completed, it will create the PeeringServerApi, do stuff with it, and then
  // pass it off via the NetworkGameJoin or NetworkGameCreate calls
  // This function will join the game with the specified name if it exists, otherwise it will create a game
  // with the specified name using the xmlFilename to setup the initial state of the world
  virtual void HackStartupNetwork(const char *playerName, const char *gameName, const char *xmlFilename, const char *recordFilename, const char *playbackFilename) = 0;

  virtual void ScriptEnableDebugging(){};
  virtual void CheatKey( KeyCode key ) = 0;
  virtual int GetShowWaterInfo() = 0;
  virtual int GetShowPlayerAnims() = 0;
  virtual void SetNumLocalPlayers(int numplayers){};
  virtual int  GetNumLocalPlayers(){return 1;};
  virtual void SetFluidLevel( const char * FluidName, float fPercentTarget, float fTimeSeconds ) = 0;


  // add parameters to the SyParameterValidator that this object is interested in
  virtual void InitParameterValidator( SyParameterValidator* parameterValidator ) const = 0;

  //============================================================================
  // Design Settings
  //============================================================================

  // Sets whether to log combat events and calculations to a file.
  virtual void SetLogMode(bool on)=0;
  virtual bool GetLogMode()=0;

  // Sets whether player is invulnerable or not
  virtual void SetGodMode(bool on)=0;
  virtual bool GetGodMode()=0;

  // Sets whether player is invulnerable or not
  virtual void SetShowDamageFlash(bool on)=0;
  virtual bool GetShowDamageFlash()=0;

  // Sets whether player is invulnerable or not
  virtual void SetUnlockAllAbilities(bool on)=0;
  virtual bool GetUnlockAllAbilities()=0;

  // Sets whether to log combat events and calculations to a file.
  virtual void SetInvisibleMode(bool on)=0;
  virtual bool GetInvisibleMode()=0;

  virtual void SetSuperSpeedMode( bool ) = 0;
  virtual bool GetSuperSpeedMode() const = 0;

  // set whether or not game is paused...
  virtual void PauseGame()=0;
  virtual void UnpauseGame()=0;

  // set time scale (for slow mo mode)
  virtual void SetTimeScale(float scale) = 0;
  virtual float GetTimeScale() = 0;

  virtual bool GetDebugLabel() = 0;
  virtual void SetDebugLabel( bool value ) = 0;

  virtual void OverloadPlayerModel(const char *modelname){};

  // Warp to a new level
  virtual void SetTransitionLevel(const char *newlevel,const char *markername) = 0;
  virtual bool IsTransitionLevelSet() const = 0;

  virtual void ReverseCamera(bool bReverse) const {};

  //============================================================================
  // Player Inventory
  //============================================================================

  //============================================================================
  // Player Inventory
  //============================================================================

  virtual int           InventoryGetItemCount( int player ) = 0;
  virtual bool          InventoryGetItemInfo(int player, int item, TiCharacterItemInfo* pInfo) = 0;
  virtual int           InventoryGetEquipSlot( int player, int item ) = 0;
  virtual bool          InventoryIsEquipped( int player, int item ) = 0;
  virtual bool          InventoryEquip( int player, int item ) = 0;
  virtual void          InventoryUnequip( int player, int item ) = 0;
  virtual bool          InventoryDrop( int player, int item ) = 0;
  virtual bool			    InventoryGetIsEnhancementValidInSlot( int player, int item, int slot ) = 0;
  virtual int           InventoryGetEnhancementSlotType (int player, int item) = 0;
  virtual void          InventoryUseEnhancement(int player, int item, int slotType, bool bApplyCost) = 0;
  virtual int           InventoryGetWeaponEnhancementItem (int player, int slotType) = 0;
  virtual void	        InventoryClearEnhancement( int player, int slotType ) = 0;
  virtual void          InventoryConvertItemToEssence(int player, int item) = 0;

  //============================================================================
  // Merchant Inventory
  //============================================================================

  virtual int           MerchantGetItemCount() = 0;
  virtual bool          MerchantGetItemInfo(int item, TiCharacterItemInfo* pInfo) = 0;
  virtual void          MerchantBuyItem(int player, int item) = 0;

  //============================================================================
  // Save/Load Games
  //============================================================================
  virtual int           SaveGameGetCount() = 0;
  virtual bool          SaveGameGetInfo(int index, TitanSave* pInfo) = 0;
  virtual bool          SaveGameLoad(int index) = 0;
  virtual bool          SaveGameOverwrite(int index, const SyString& desc) = 0;
  virtual bool          SaveGameCreate(const SyString& desc) = 0;
  virtual bool          SaveGameOverwriteLast() = 0;
  virtual bool          SaveGameLoadLast() = 0;

  //============================================================================
  // Player Spells/Abilities
  //============================================================================

  // all of these are going away away away
  virtual int           AbilityMappedGetIndex( int player, int button ) = 0;

  virtual int           AbilityCurrentGetCount( int player ) = 0;
  virtual bool          AbilityCurrentGetName( int player, int index, SyString& name) = 0;
  virtual bool          AbilityCurrentGetDescription( int player, int index, SyString& desc ) = 0;
  virtual bool          AbilityCurrentIsActive( int player, int index ) = 0;  // true if it is mappable
  virtual void          AbilityCurrentMap( int player, int index, int button ) = 0;

  virtual int           AbilityAvailableGetCount( int player ) = 0;
  virtual bool          AbilityAvailableGetName( int player, int index, SyString& name ) = 0;
  virtual int           AbilityAvailableGetCost( int player, int index ) = 0;
  virtual bool          AbilityAvailableGetDescription( int player, int index, SyString& desc ) = 0;
  virtual bool          AbilityAvailableIsBuyable( int player, int index ) = 0; // for available abilities that you just don't have enough skill points for (greyed/disabled)
  virtual void          AbilityAvailableBuy( int player, int index ) = 0;

  // Added by Wiesman
  virtual int           AbilityGetNumPoints( int player ) = 0;
  virtual bool          AbilityGetName(int32 iPlayer, SyResourceID idAbility, SyString &strName) = 0;
  virtual bool          AbilityGetDescription(int32 iPlayer, SyResourceID idAbility, SyString &strDescr) = 0;
  virtual SyResourceID  AbilityGetIcon(int32 iPlayer, SyResourceID idAbility) = 0;
  virtual int32         AbilityGetRank(int32 iPlayer, SyResourceID idAbility) = 0;
  virtual void          AbilityMapButton(int32 iPlayer, SyResourceID idAbility, int32 iButton) = 0;
  virtual int32         AbilityGetButtonForAbility(int32 iPlayer, SyResourceID idAbility) = 0;
  virtual SyResourceID  AbilityGetAbilityForButton(int32 iPlayer, int32 iButton) = 0;
  virtual void          AbilityBuy(int iPlayer, SyResourceID idAbility) = 0;


  //============================================================================
  // Scene options
  //============================================================================

  // Gets or sets a value that indicates whether the bloom post processing effect is enabled
  virtual bool GetBloom() = 0;
  virtual void SetBloom( bool value ) = 0;

  virtual bool GetMultiPass() = 0;
  virtual void SetMultiPass( bool value ) = 0;

  virtual void SetNoWaterRayCasts( bool value ) = 0;
  virtual void SetNoPlunk( bool value ) = 0;

  enum DrawMode
  {
    DrawModeNormal = 0,
    DrawModeWire   = 1,
    DrawModeSolid  = 2,
  };
  virtual DrawMode GetDrawMode() = 0;
  virtual void SetDrawMode( DrawMode value ) = 0;


  //============================================================================
  // Performance Stats
  //============================================================================

  virtual int GetFPS() = 0;
  virtual int GetBatches() = 0;
  virtual int GetVertexes() = 0;
  virtual int GetTriangles() = 0;
  virtual uint32 GetPixels() = 0;
  virtual int GetDegenerateTriangles() = 0;

  virtual int GetAverageFPS() = 0;
  virtual int GetAverageBatches() = 0;
  virtual int GetAverageVertexes() = 0;
  virtual int GetAverageTriangles() = 0;
  virtual int GetAverageDegenerateTriangles() = 0;


  //============================================================================
  // Debug Methods (Jon Wiesman)
  //============================================================================

  virtual int Debug_AddItemToPlayerInventory( int32 nPlayerId, const char8 *pszItem ) = 0;

  //============================================================================
  // Player Stats
  //============================================================================

  virtual int   GetPlayerName( int32 PlayerId, char8* pBuffer, int32 BuffLen ) const = 0;
  virtual void  SetPlayerName( int32 PlayerId, const char8* pName ) = 0;

  virtual int GetPlayerHealthMax( int32 PlayerId ) = 0;
  virtual int GetPlayerHealth( int32 PlayerId ) = 0;
  virtual int GetPlayerBlockMax( int32 PlayerId ) = 0;
  virtual int GetPlayerBlock( int32 PlayerId ) = 0;
  virtual int GetPlayerManaMax( int32 PlayerId ) = 0;
  virtual int GetPlayerMana( int32 PlayerId ) = 0;
  virtual int GetPlayerEssence( int32 PlayerId ) = 0;
  virtual int GetPlayerLevel( int32 PlayerId ) = 0;
  virtual int GetPlayerCurExp( int32 PlayerId ) = 0;
  virtual int GetPlayerNextLevelExp( int32 PlayerId ) = 0;
  virtual int GetPlayerPrevLevelExp( int32 PlayerId ) = 0;
  virtual int GetPlayerLocation( int32 iPlayer, SyVect3 &vecLocation ) = 0;
  virtual float GetPlayerHeading( int32 iPlayer ) const { return 0.0f; } 
  virtual void SetPlayerCharacter(int iPlayer, const char* characterMasterName) = 0;
  virtual float GetCameraHeading() { return 0.0f; }

  // Begin Added by Jon Wiesman
  virtual int32 GetPlayerAttackPower(int32 iPlayer) const {return 0;}
  virtual int32 GetPlayerMagicPower(int32 iPlayer) const {return 0;}
  virtual int32 GetPlayerMeleeDefense(int32 iPlayer) const {return 0;}
  virtual int32 GetPlayerSpellDefense(int32 iPlayer) const {return 0;}
  virtual int32 GetPlayerBonusPoints(int32 iPlayer) const {return 0;}
  virtual void AddHealthPoints(int32 iPlayer, int32 nPoints) {}
  virtual void AddManaPoints(int32 iPlayer, int32 nPoints) {}
  virtual void AddAttackPowerPoints(int32 iPlayer, int32 nPoints) {}
  virtual void AddMagicPowerPoints(int32 iPlayer, int32 nPoints) {}
  virtual void AddMeleeDefensePoints(int32 iPlayer, int32 nPoints) {}
  virtual void AddSpellDefensePoints(int32 iPlayer, int32 nPoints) {}
  // End Added by Jon Wiesman
  virtual void          SetUnfocused(bool unfocus) = 0;

  virtual int GetDifficultySetting() = 0;
  virtual void SetDifficultySetting(int ds) = 0;

  //============================================================================
  // NPC Stats
  //============================================================================
  virtual bool GetFirstNPCInfo(int player, int& health, int& healthmax, int& block, int& blockmax, int& x, int& y, bool& offscreen, bool& isBlocking) = 0;
  virtual bool GetNextNPCInfo(int player, int& health, int& healthmax, int& block, int& blockmax, int& x, int& y, bool& offscreen, bool& isBlocking) = 0;

  //============================================================================
  // UI Activate String
  //============================================================================

  virtual bool GetActivateString(int playerId, char *buffer,int buffLen){return false;};

  //============================================================================
  // Subtitles
  //============================================================================

  virtual bool GetSubtitleText(char *buffer,int buffLen){return false;};
  virtual void SetSubtitleText(char *buffer,float time){};

  //============================================================================
  // Localized Text
  //============================================================================

  // main translation methods
  virtual void  InitT4WithLanguage(int32 iLanguage) = 0;
  virtual int32 GetT4Language() const = 0;
  virtual int32 T4Lookup( char8* pBuffer, int BufLen, uint32 StringId ) = 0;
  virtual int32 T4Expand( char8* pBuffer, int BufLen, uint32 StringId, float32 Faction = 0.5f ) = 0;

  // support methods
  virtual int32 T4SetTopic( char8 Slot ) = 0;
  virtual int32 T4SetContext( char8 Slot,
                            uint32 StringId,
                            int Gender = 0,
                            int Count = 1,
                            int BeingType = 0,
                            int RespectType = -3,
                            int Age = 0 ) = 0;
  virtual int32 T4SetFinal( char8 Slot,
                          const char8* pText,
                          int Gender = 0,
                          int Count = 1,
                          int BeingType = 0,
                          int RespectType = -3,
                          int Age = 0 ) = 0;
};

/**/

#endif // __TITANI_H__

