/******************************************************************
  
  Module:  gameerror.h
  
  Author: Borut Pfeifer
  
  Description: channel based assert/warning logging system for 
               data driven game engine

  
  Copyright 2005 Sony Online Entertainment.  All rights reserved.
  
*******************************************************************/

#ifndef GAMEERROR_H
#define GAMEERROR_H

//-------------------------------------------------------- Includes
//-------------------------------------------------------- Typedefs
//---------------------------------------------- Class Declarations

//#ifdef _DEBUG
#define _GAME_ERROR_REPORTING
//#endif

void _GameErrorReportFatal(const char* channel, bool bCondition, const char* format, ...);
void _GameErrorReportAssert(const char* channel, bool bCondition, const char* format, ...);
void _GameErrorReportWarning(const char* channel, bool bCondition, const char* format, ...);

#ifdef _GAME_ERROR_REPORTING

extern char* __GameErrorReport_file;
extern int __GameErrorReport_line;
extern bool* __GameErrorReport_skip;

#define GAME_FATAL __GameErrorReport_file=__FILE__;__GameErrorReport_line=__LINE__;__GameErrorReport_skip=NULL;_GameErrorReportFatal
#define GAME_ASSERT __GameErrorReport_file=__FILE__;__GameErrorReport_line=__LINE__;{static bool sbGRSkip = false; __GameErrorReport_skip = &sbGRSkip;} _GameErrorReportAssert
#define GAME_WARN __GameErrorReport_file=__FILE__;__GameErrorReport_line=__LINE__;{static bool sbGRSkip = false; __GameErrorReport_skip = &sbGRSkip;} _GameErrorReportWarning

#define GAME_ERROR_ENABLECHANNEL(channelName, bEnable) cGameErrorLog::Get()->EnableChannel((channelName), (bEnable))
#define GAME_ERROR_PROMOTE_TO_FATAL(channelName) cGameErrorLog::Get()->PromoteToFatal(channelName)
#define GAME_ERROR_PROMOTE_TO_ASSERT(channelName) cGameErrorLog::Get()->PromoteToAssert(channelName) 
#define GAME_ERROR_DEMOTE_TO_WARNING(channelName) cGameErrorLog::Get()->DemoteToWarning(channelName)

#else

#define GAME_FATAL  if (0) _GameErrorReportFatal
#define GAME_ASSERT if (0) _GameErrorReportAssert
#define GAME_WARN if (0) _GameErrorReportWarning

#define GAME_ERROR_ENABLECHANNEL(channelName, bEnable)

#define GAME_ERROR_PROMOTE_TO_FATAL(channelName)
#define GAME_ERROR_PROMOTE_TO_ASSERT(channelName)
#define GAME_ERROR_DEMOTE_TO_WARNING(channelName) 

#endif


// standard error channels
#define ERROR_DESIGN "DESIGN"
#define ERROR_ART "ART"
#define ERROR_CODE "CODE"


#ifdef _GAME_ERROR_REPORTING

class SyInputConsole;
class SyRaster;
class SyColor32F;
class SyScene;

class cGameErrorLog
{
public:
  cGameErrorLog();
  ~cGameErrorLog();
  
  static cGameErrorLog* Get() { return smpThis; }

  void Init(SyInputConsole* pConsole, SyRaster* pRaster, SyScene* pScene);

  void SetChannelBackgroundColor(const char* channelName, const SyColor32F& color);
  void PromoteToFatal(const char* channelName);
  void PromoteToAssert(const char* channelName);
  void DemoteToWarning(const char* channelName);

private:
  friend void _GameErrorReportFatal(const char* channel, bool bCondition, const char* format, ...);
  friend void _GameErrorReportAssert(const char* channel, bool bCondition, const char* format, ...);
  friend void _GameErrorReportWarning(const char* channel, bool bCondition, const char* format, ...);

  static cGameErrorLog* smpThis;

  class Implementation;

  Implementation* mpImp;
};

#endif

//------------------------------------------- Function Declarations
//--------------------------------------------------------- Globals
//------------------------------------------------ Inline Functions

#endif
