/******************************************************************
  
  Module:  levelobj.h
  
  Author: Sean Craig
  
  Description: the level object, which contains global information
  about the game which should be transmitted across the networking
  using the PropObj system.

  There is only one level object in a game, and it is owned by a 
  single client.

  
  Copyright 2005 Sony Online Entertainment.  All rights reserved.
  
*******************************************************************/

#ifndef LEVEL_OBJ_H
#define LEVEL_OBJ_H

//-------------------------------------------------------- Includes
#include "gameobj.h"
#include "SyLoader.h"
#include "SyMap.h"
#include "SyString.h"
#include "SySceneTypes.h"
#include "quest.h"
//-------------------------------------------------------- Typedefs
//---------------------------------------------- Class Declarations
class SyLoader;
class cAIBlackboard;
class cAreaEffectSys;
class cScriptInstanceDataProxy;
class cTreasureSet;

static const int LEVELOBJECT_CLASSID         =  0xFFFFA002;
class cLevelObject : public SyPropObject
{
public: 
  cLevelObject();
  ~cLevelObject();

  /* Property Class Support */
  static int           InitPropClass();
  static SyPropObject* Creator();  

  const char *      GetLevelName() const {return mLevelName.AsChar();}
  void              SetLevelName(const char *name);

  const char *      GetCameraName() const {return mCameraName.AsChar();}
  void              SetCameraName(const char *name);

  const char *      GetModelName() const {return mModelName.AsChar();}
  void              SetModelName(const char *name);
  void              Init(Titan *titan);
  void              Exit();

  Titan*            GetTitan() const {return mpTitan;}
  tGameID           GetID() const {return mID;}

  void              Update(float time);

  void              NetworkReceiveBroadcast(const char *state, int maxlen);
  void              NetworkReceiveMessage(const char *state, int maxlen);

  void              SetRemote(bool bRemote);
  bool              IsRemote(){return !IsLocal();};
  bool              IsLocal();
  int32             PlayMusic(const char *filename, bool bLooping);  // returns resource handle.  Registers& loads resource
  void              StopMusic();

  void              SetQuest(tQuestID id,int32 value);
  int32             GetQuest(tQuestID id);
  void              GetHighlightMaterial(int32 &handle,SyVect2 &minUV,SyVect2 &maxUV);

  void              DisableScriptCam();
  void              EnableScriptCam(int               scriptCamSetting,
                                    tGameObjectID     scriptCamSourceID,
                                    tGameObjectID     scriptCamTargetID,
                                    float             scriptCamSourceYOffset,
                                    float             scriptCamTargetYOffset,
                                    float             scriptCamViewAngle);
  
  void              SetEnableInput(bool bEnableInput);
  void              SetNumGameObjects(int num); 
  int               GetNumGameObjects() const { return mNumGameObjects; }

  cTreasureSet*     GetDefaultTreasureSet() const;

  tGameID           GetLevelNameHashID() const { return mLevelNameHashID; }

  void              SaveScriptData();
  void              ClearScriptData();

  static const int LEVEL_OBJECT_GAME_ID = 0x1;

  static const SyPropID PropId_ModelName     = 0x0000;
  static const SyPropID PropId_CameraName    = 0x0001;
  static const SyPropID PropId_AIBlackboard  = 0x0002;
  static const SyPropID PropId_LoadedSounds  = 0x0003;
  static const SyPropID PropId_LevelName     = 0x0004;
  static const SyPropID PropId_AreaEffectSys = 0x0005;
  static const SyPropID PropId_MusicName     = 0x0006;
  static const SyPropID PropId_QuestSys      = 0x0007;

  static const SyPropID PropId_ScriptCamMode            = 0x0008;
  static const SyPropID PropId_ScriptCamSetting         = 0x0009;
  static const SyPropID PropId_ScriptCamSourceID        = 0x000A;
  static const SyPropID PropId_ScriptCamTargetID        = 0x000B;
  static const SyPropID PropId_ScriptCamSourceYOffset   = 0x000C;
  static const SyPropID PropId_ScriptCamTargetYOffset   = 0x000D;
  static const SyPropID PropId_ScriptCamViewAngle       = 0x000E;

  static const SyPropID PropId_NumGameObjects           = 0x000F;
  static const SyPropID PropId_EnableInput              = 0x0010;
  static const SyPropID PropId_ScriptDataProxies        = 0x0011;

protected:

  void              _LoadHighlightMaterial();
  void              UpdateLevelNameHashID();

  tGameID           mID; // used to synch up across network..
  bool              mRemote;
  SyString          mModelName;  // the level model, geometry + textures for the world
  SyString          mCameraName;
  SyString          mLevelName;
  tGameID           mLevelNameHashID;
  Titan *           mpTitan;
  SyLoader*         mpLoader;
  cAIBlackboard*    mpAIBlackboard;
  cAreaEffectSys*   mpAreaEffectSys;
  cQuestSys*        mpQuestSys;

  // data to handle cutscene info when we join a net game
  bool              mbScriptCamMode;
  int               mScriptCamSetting;
  tGameObjectID     mScriptCamSourceID;
  tGameObjectID     mScriptCamTargetID;
  float             mScriptCamSourceYOffset;
  float             mScriptCamTargetYOffset;
  float             mScriptCamViewAngle;

  int               mNumGameObjects;
  bool              mbEnableInput;

  SyString          mMusicName;
  int32             mMusicID;

  int32             mHighlightHandle; 
  SyVect2           mHighlightMinUV;
  SyVect2           mHighlightMaxUV;

  SyVector<cScriptInstanceDataProxy*>  mScriptDataProxies;
};

//------------------------------------------- Function Declarations
//--------------------------------------------------------- Globals
void RegPropClasses_LevelObj();

//------------------------------------------------ Inline Functions

#endif
