/********************************************************************

  Module: TiGameWindow

  Author: Carlo Ordonez

  Description:

    Game UI window

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiGameWindow.h"
#include "TiUI.h"
#include "TiPauseMenu.h"
#include "TiCharacterWindow.h"
#include "TiActivateWindow.h"
#include "TiPawnOutput.h"
#include "TiPawnDebug.h"
#include "TiCharacterData.h"
#include "TiCommandMessage.h"
#include "InputConsole.h"
#include "TitanI.h"
#include "TiTestDialog.h"
#include "SyDebug.h"
#include "TiMessageBox.h"
#include "SyBuildVersion.h"
#include "SyRaster.h"
#include "SyScene.h"
#include "TiUILayout.h"
#include "SyESFParse.h"

//***********************************************************************
// Constructor/Destructor
//***********************************************************************

TiGameWindow::TiGameWindow( TiApp* pApp, TitanUII *pTitanUI, TitanI *pTitan )
: TiTopLevelWindow( pApp ), mpTitanUI(pTitanUI), mpTitan(pTitan)
{
  // at least one player
  mPlayerCount = 1;

  for( int32 i = 0; i < PLAYER_MAX; i++ )
  {
    mPlayerStats[i].mHealth     = 0;
    mPlayerStats[i].mHealthMax  = 1;
    mPlayerStats[i].mMana       = 0;
    mPlayerStats[i].mManaMax    = 1;
    mPlayerStats[i].mXP         = 0;
    mPlayerStats[i].mXPMax      = 1;
    mPlayerStats[i].mLevel      = 0;
    mPlayerStats[i].mName[0]    = 0;

    mPlayerStats[i].mPrevHealth = 1;
    mPlayerStats[i].mPrevMana   = 1;
    mPlayerStats[i].mHit        = false;
    mPlayerStats[i].mPrevHit    = false;
  }

  mpPauseMenu = SyNew TiPauseMenu( this );
  mpPauseMenu->Show( false );

  mpCharacterData = SyNew TiCharacterData();
  mpCharacterWindow = SyNew TiCharacterWindow( this, mpCharacterData );
  mpCharacterWindow->Show( false );

  mpActivateWindow = SyNew TiActivateWindow( this );
  mpActivateWindow->Show( true );

  mpPawnOutput = SyNew TiPawnOutput( this );
  mpPawnOutput->Show( false );

  mpPawnDebug = SyNew TiPawnDebug( this );
  mpPawnDebug->Show( false );

  mpNPCLayout = SyNew TiUILayout();
  mpNPCLayout->Init( "npc_health.uiml" );

  mpPlayerLayout = SyNew TiUILayout();
  mpPlayerLayout->Init( "player_hud.uiml" );

  m_sprCharacterFrame.SetInfo( "resources_frame.tga" );
  m_sprExpFrame.SetInfo( "resources_exp_frame.tga" );
  m_sprHealthBar.SetInfo( "resources_resources_health.tga" );
  m_sprManaBar.SetInfo( "resources_resources_mana.tga" );

  const TiRect c_rectExpDiscs[PLAYER_MAX] = {
    TiRect(TiPoint(75, 37), TiSize(59, 59)),
    TiRect(TiPoint(275, 37), TiSize(59, 59)),
    TiRect(TiPoint(475, 37), TiSize(59, 59)),
    TiRect(TiPoint(675, 37), TiSize(59, 59)),
  };

  for(int i = 0; i < PLAYER_MAX; i++)
  {
    m_cdExpBars[i].Init(c_rectExpDiscs[i], "resources_exp.tga");
  }

  m_sprPortraits[0].SetInfo("resources_portrait_brute_red.tga");
  m_sprPortraits[1].SetInfo("resources_portrait_brute_yellow.tga");
  m_sprPortraits[2].SetInfo("resources_portrait_brute_blue.tga");
  m_sprPortraits[3].SetInfo("resources_portrait_brute_green.tga");
}

TiGameWindow::~TiGameWindow()
{
  delete mpCharacterData;
  delete mpNPCLayout;
  delete mpPlayerLayout;
}

//***********************************************************************
// DrawCharacterHud
//***********************************************************************

void TiGameWindow::DrawCharacterHud(int iCharacter, float fHealth, float fMana, float fExp)
{
  const TiPoint arChars[PLAYER_MAX] = {
    TiPoint(80, 30),
    TiPoint(280, 30),
    TiPoint(480, 30),
    TiPoint(680, 30),
  };

  TiPoint ptOrigin = arChars[iCharacter];

  const TiPoint c_ptHealthOffset(39, 6);
  const TiPoint c_ptManaOffset(44, 16);
  const TiPoint c_ptExpFrameOffset(-8, 5);
  const TiPoint c_ptPortraitOffset(3, 15);

  // first draw the exp frame
  m_sprExpFrame.Draw(ptOrigin + c_ptExpFrameOffset);
  
  // then draw the exp disc
  const float32 c_fExpRadiansStart = (SY_PI * 2.f) * (72.f / 360.f);
  const float32 c_fExpRadiansEnd = (SY_PI * 2.f) * (385.f / 360.f);
  m_cdExpBars[iCharacter].SetAngles(c_fExpRadiansStart, c_fExpRadiansStart + fExp * (c_fExpRadiansEnd - c_fExpRadiansStart));
  m_cdExpBars[iCharacter].Draw();

  // then draw the big frame
  m_sprCharacterFrame.Draw(ptOrigin);

  // the draw health bar
  TiRect rectHealthClip(ptOrigin + c_ptHealthOffset, m_sprHealthBar.GetSize());
  rectHealthClip.right = rectHealthClip.left + (int)((float)rectHealthClip.Width() * fHealth);
  m_sprHealthBar.Draw(ptOrigin + c_ptHealthOffset, rectHealthClip);

  // the mana health bar
  TiRect rectManaClip(ptOrigin + c_ptManaOffset, m_sprManaBar.GetSize());
  rectManaClip.right = rectManaClip.left + (int)((float)rectManaClip.Width() * fMana);
  m_sprManaBar.Draw(ptOrigin + c_ptManaOffset, rectManaClip);

  // finally, draw the portrait
  m_sprPortraits[iCharacter].Draw(ptOrigin + c_ptPortraitOffset);
}


//***********************************************************************
// DrawSelf
//***********************************************************************

void TiGameWindow::DrawSelf()
{
  if( !mpTitanUI->GetShowUI() )
  {
    mpActivateWindow->SetSubtitleText("");
    for( int p = 0; p < GetPlayerCount(); p++ )
    {
      mpActivateWindow->SetPlayerText( p, "" );
    }
    return;
  }

  int32 screenW, screenH;
  TiUI::Instance()->GetDisplaySize( &screenW, &screenH );

  TiUILayoutItem info;

  for(int i = 0; i < GetPlayerCount(); i++)
  {
    const Stats& statPlayer = mPlayerStats[i];
    float32 fHealth = (float32)statPlayer.mHealth / (float32)statPlayer.mHealthMax;
    float32 fMana = (float32)statPlayer.mMana / (float32)statPlayer.mManaMax;
    float32 fExp = (float32)statPlayer.mXP / (float32)statPlayer.mXPMax;

    DrawCharacterHud(i, fHealth, fMana, fExp);
  }

  SyVect3 playerPos;
  float   cameraHeading;
  mpTitan->GetPlayerLocation(0, playerPos);
  cameraHeading = mpTitan->GetCameraHeading();
  m_MiniMap.Update(playerPos, cameraHeading);
  m_MiniMap.Draw();

#if 0
  // health bars
  SyVect2I offset, size;

  mpPlayerLayout->GetLayoutInfo( "frame", &info );
  TiUI::Instance()->GetBitmapInfo( info.mFilename.AsChar(), offset, size );
  int32 frameW = size.X;

  int32 w = screenW - 150;

  // spacing between players
  int32 incr = 0;
  switch( GetPlayerCount() )
  {
  case 2:
    incr = w - (frameW);
    break;
  case 3:
    incr = (w/2) - (frameW/2);
    break;
  case 4:
    incr = (w/3) - (frameW/2) + 40;
    break;
  case 1:
  default:
    break;    
  }

  // player bars
  SyVect2I loc( 75, 75 );
  float32 fade = .8f;
  for( int32 i = 0; i < GetPlayerCount(); i++ )
  {
    SyVect2I imgSize;
    SyVect2I pt = loc;

    Stats& player = mPlayerStats[i];

    // frame
    mpPlayerLayout->GetLayoutInfo( "frame", &info );
    TiUI::Instance()->GetBitmapInfo( info.mFilename.AsChar(), offset, size );
    pt.Y += 2;
    offset.Y += 2;
    size.Y -= 2;
    TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, pt, offset, size );

    // portrait
    mpPlayerLayout->GetLayoutInfo( "portrait", &info );
    pt = loc + info.mOffset;
    TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, pt );

    // experience, part 1
    float32 fXP = (float32)player.mXP / (float32)player.mXPMax;

    /* 
    if( fXP > .125f )
    {
      mpPlayerLayout->GetLayoutInfo( "glow1", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_VERTICAL, loc + info.mOffset );
      mpPlayerLayout->GetLayoutInfo( "spike1", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_VERTICAL, loc + info.mOffset );
    }
    if( fXP > .25f )
    {
      mpPlayerLayout->GetLayoutInfo( "glow2", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
      mpPlayerLayout->GetLayoutInfo( "spike2", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
    }
    if( fXP > .375f )
    {
      mpPlayerLayout->GetLayoutInfo( "glow3", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
      mpPlayerLayout->GetLayoutInfo( "spike3", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
    }
    if( fXP > .5f )
    {
      mpPlayerLayout->GetLayoutInfo( "glow4", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
      mpPlayerLayout->GetLayoutInfo( "spike4", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
    }
    if( fXP > .625f )
    {
      mpPlayerLayout->GetLayoutInfo( "glow5", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_HORIZONTAL, loc + info.mOffset );
      mpPlayerLayout->GetLayoutInfo( "spike5", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_HORIZONTAL, loc + info.mOffset );
    }
    if( fXP > .75f )
    {
      mpPlayerLayout->GetLayoutInfo( "glow6", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_HORIZONTAL, loc + info.mOffset );
      mpPlayerLayout->GetLayoutInfo( "spike6", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_HORIZONTAL, loc + info.mOffset );
    }
    if( fXP > .875f )
    {
      mpPlayerLayout->GetLayoutInfo( "glow7", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_BOTH, loc + info.mOffset );
      mpPlayerLayout->GetLayoutInfo( "spike7", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_BOTH, loc + info.mOffset );
    }
    //*/
    /*
    if( true )
    {
      mpPlayerLayout->GetLayoutInfo( "glow8", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_VERTICAL, loc + info.mOffset );
      mpPlayerLayout->GetLayoutInfo( "spike8", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_VERTICAL, loc + info.mOffset );
    }
    //*/

    // lens
    mpPlayerLayout->GetLayoutInfo( "lens", &info );
    TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );

    // health transition
    mpPlayerLayout->GetLayoutInfo( "health", &info );
    TiUI::Instance()->GetBitmapInfo( info.mFilename.AsChar(), offset, size );
    float32 f = (float32)player.mPrevHealth / (float32)player.mHealthMax;
    int32 barWidth = size.X;
    int32 w = int32((float32)barWidth * f);
    imgSize = SyVect2I( w, size.Y );
    TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset, offset, imgSize, SYALPHABLENDMODE_AVG, .4f );

    //TiUI::Instance()->GetBitmapInfo( pMask, offset, size );
    //pt.X += imgSize.X - size.X;
    //TiUI::Instance()->Blit( pMask, 0, pt );

    // health
    float32 healthFade = fade;
    if( player.mHit && !player.mPrevHit )
    {
      healthFade = 1.f;
      player.mPrevHit = true;
    }
    barWidth = size.X;
    f = (float32)player.mHealth / (float32)player.mHealthMax;
    w = int32((float32)barWidth * f);
    imgSize = SyVect2I( w, size.Y );
    TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset, offset, imgSize, SYALPHABLENDMODE_AVG, healthFade );

    // mana transition
    mpPlayerLayout->GetLayoutInfo( "mana", &info );
    TiUI::Instance()->GetBitmapInfo( info.mFilename.AsChar(), offset, size );
    barWidth = size.X;
    f = (float32)player.mPrevMana / (float32)player.mManaMax;
    w = int32((float32)barWidth * f);
    imgSize = SyVect2I( w, size.Y );
    TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset, offset, imgSize, SYALPHABLENDMODE_AVG, .4f );

    //TiUI::Instance()->GetBitmapInfo( pMask, offset, size );
    //pt.X += imgSize.X - size.X;
    //TiUI::Instance()->Blit( pMask, SYRASTER_BLITFLAG_FLIP_VERTICAL, pt );

    // mana
    barWidth = size.X;
    f = (float32)player.mMana / (float32)player.mManaMax;
    w = int32((float32)barWidth * f);
    imgSize = SyVect2I( w, size.Y );
    TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset, offset, imgSize, SYALPHABLENDMODE_AVG, fade );

    // highlight
    mpPlayerLayout->GetLayoutInfo( "highlight", &info );
    TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );

    // studs
    mpPlayerLayout->GetLayoutInfo( "studs", &info );
    TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );

    // experience, part 2
    if( fXP > .125f )
    {
      mpPlayerLayout->GetLayoutInfo( "gem1", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_BOTH, loc + info.mOffset ); 
    }
    if( fXP > .25f ) 
    { 
      mpPlayerLayout->GetLayoutInfo( "gem2", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
    }
    if( fXP > .375f ) 
    { 
      mpPlayerLayout->GetLayoutInfo( "gem3", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_VERTICAL, loc + info.mOffset );
    }
    if( fXP > .5f )
    {
      mpPlayerLayout->GetLayoutInfo( "gem4", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
    }
    if( fXP > .625f )
    {
      mpPlayerLayout->GetLayoutInfo( "gem5", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
    }
    if( fXP > .75f )
    {
      mpPlayerLayout->GetLayoutInfo( "gem6", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset ); 
    }
    if( fXP > .875f )
    {
      mpPlayerLayout->GetLayoutInfo( "gem7", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), SYRASTER_BLITFLAG_FLIP_HORIZONTAL, loc + info.mOffset ); 
    }
    /*
    if( true )
    {
      mpPlayerLayout->GetLayoutInfo( "gem8", &info );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, loc + info.mOffset );
    }
    //*/

    // update transitions
    if( player.mPrevHealth > player.mHealth ) {
      player.mPrevHealth--;
      player.mHit = true;
    } else {
      player.mPrevHealth = player.mHealth;
      player.mHit = false;
      player.mPrevHit = false;
    }

    if( player.mPrevMana > player.mMana ) {
      player.mPrevMana--;
    } else {
      player.mPrevMana = player.mMana;
    }
    
  }

  // NPC display
  SyVect2I offset, size;
  int32 health=0, block=0;
  int32 healthmax=0, blockmax=0;
  bool offscreen = true;
  bool isBlocking = true;
  SyVect2I npc;
  float32 fade = .7f;
  for( bool npcInfo = mpTitan->GetFirstNPCInfo(i,health,healthmax,block,blockmax,npc.X,npc.Y,offscreen,isBlocking); npcInfo;
        npcInfo = mpTitan->GetNextNPCInfo(i,health,healthmax,block,blockmax,npc.X,npc.Y,offscreen,isBlocking) )
  {
    if( !offscreen && health > 0 )
    {
      npc.X -= size.X / 2;
      SyVect2I pt;

      // health border
      mpNPCLayout->GetLayoutInfo( "healthBorder", &info );
      pt = npc + info.mOffset;
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, pt, SYALPHABLENDMODE_AVG );

      // health
      mpNPCLayout->GetLayoutInfo( "health", &info );
      TiUI::Instance()->GetBitmapInfo( info.mFilename.AsChar(), offset, size );
      pt = npc + info.mOffset;
      barWidth = size.X;
      f = (float32)health / (float32)healthmax;
      w = int32((float32)barWidth * f);
      imgSize = SyVect2I( w, size.Y );
      TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, pt, offset, imgSize, SYALPHABLENDMODE_AVG );

      if( isBlocking )
      {
        // block border
        SyVect2I blockOffset, blockSize;
        TiUILayoutItem blockInfo;
        mpNPCLayout->GetLayoutInfo( "shield", &blockInfo );
        mpNPCLayout->GetLayoutInfo( "shieldBorder", &info );
        TiUI::Instance()->GetBitmapInfo( blockInfo.mFilename.AsChar(), blockOffset, blockSize );
        TiUI::Instance()->GetBitmapInfo( info.mFilename.AsChar(), offset, size );
        pt = npc + info.mOffset;

        barWidth = blockSize.X;
        f = (float32)block / (float32)blockmax;
        w = int32((float32)barWidth * f);

        imgSize = SyVect2I( w + 2, size.Y );
        TiUI::Instance()->Blit( info.mFilename.AsChar(), 0, pt, offset, imgSize, SYALPHABLENDMODE_AVG );

        // block
        pt = npc + blockInfo.mOffset;
        imgSize = SyVect2I( w, blockSize.Y );
        TiUI::Instance()->Blit( blockInfo.mFilename.AsChar(), 0, pt, blockOffset, imgSize, SYALPHABLENDMODE_AVG );
      }
    }
  }

  // next player
  loc.X += incr;
#endif
}

//***********************************************************************
// ProcessKeyDown
//***********************************************************************

bool TiGameWindow::ProcessKeyDown( int32 KeyCode )
{
  switch( KeyCode )
  {
  case SyInputEvent::KEYCODE_F2:
  case SyInputEvent::KEYCODE_ESCAPE:
    if( this == FindFocus() )
    {
      OpenPauseMenu(0);
    }
  	return( true );

  case SyInputEvent::KEYCODE_F1:
    if( this == FindFocus() )
    {
      OpenCharacterWindow(0);
    }
    return( true );

#if 0
  case SyInputEvent::KEYCODE_F8:
    {
      SyDebug( "KEYCODE_F8\n" );
      TiTestDialog test( this, TiUI::Instance() );
      int result = test.ShowModal();
      SyDebug( "ShowModal() == %i\n", result );
    }
    return( true );
#endif

  default:
    return( TiWindow::ProcessKeyDown( KeyCode ) );
  }
}

//***********************************************************************
// ProcessButtonDown
//***********************************************************************

bool TiGameWindow::ProcessButtonDown( int32 DeviceId, int32 ButtonId )
{
  switch( TiWindow::GetButton( ButtonId ) )
  {
  case TiButtonMap::kSelect:
    if( !mpTitanUI->IsLetterboxMode() && this == FindFocus() )
    {
      OpenCharacterWindow( DeviceId );
      return( true );
    }
    break;

  case TiButtonMap::kStart:
    if( !mpTitanUI->IsLetterboxMode() && this == FindFocus() )
    {
      OpenPauseMenu( DeviceId );
      return( true );
    }
    break;

  default:
    break;
  }

  return( TiWindow::ProcessButtonDown( DeviceId, ButtonId ) );
}

//***********************************************************************
// ProcessMessage
//***********************************************************************

void TiGameWindow::ProcessMessage( TiWindowMessage* pMessage )
{
  switch( pMessage->GetMessageId() )
  {
  case TIWINDOW_CLOSE:
    {
      // close window & return to game
      TiWindow* pWindow = pMessage->GetSender();
      pWindow->Show( false );
      if( this->IsVisible() )
      {
        this->SetFocus();
      }
      mpActivateWindow->Show( true );
      mpTitan->UnpauseGame();
      mpTitan->SetUnfocused(false);
    }
    break;

  case TIWINDOW_COMMAND:
    {
      TiCommandMessage* pMsg = static_cast<TiCommandMessage*>(pMessage);
      if( pMsg->GetCommandId() == TI_CMD_BUILD_INFO )
      {
        TiMessageBox mbox( mpPauseMenu, g_buildVersionString, "Build Number" );
        mbox.ShowModal();
      }
      else
      {
        mpTitanUI->ProcessCommandMessage( pMsg );
      }
    }
    break;

  default:
    TiWindow::ProcessMessage( pMessage );
    break;
  }
}

//***********************************************************************
// ProcessReloadLayout
//***********************************************************************

void TiGameWindow::ProcessReloadLayout()
{
  mpNPCLayout->Reload();
  mpPlayerLayout->Reload();
}

//***********************************************************************
// OpenCharacterWindow
//***********************************************************************

void TiGameWindow::OpenCharacterWindow( int32 Player )
{
  mpTitan->PauseGame();
  mpTitan->SetUnfocused(true);
  mpActivateWindow->Show( false );
  mpCharacterData->Init( mpTitan );
  mpCharacterData->Update();
  mpCharacterWindow->Init( GetPlayerCount(), Player );
  mpCharacterWindow->Show( true );
  mpCharacterWindow->SetFocus();
}

//***********************************************************************
// OpenPauseMenu
//***********************************************************************

void TiGameWindow::OpenPauseMenu( int32 Player )
{
  TiUI::Instance()->PlaySound( TiUI::kSoundName );
  mpTitan->PauseGame();
  mpPauseMenu->Init( Player, mpTitanUI->GetOptions() );

  // set checkable entries
  mpPauseMenu->SetCheck( TI_CMD_INVISIBLE,      mpTitan->GetInvisibleMode() );
  mpPauseMenu->SetCheck( TI_CMD_GOD_MODE,       mpTitan->GetGodMode() );
  mpPauseMenu->SetCheck( TI_CMD_LOG,            mpTitan->GetLogMode() );
  mpPauseMenu->SetCheck( TI_CMD_DEBUG_LABEL,    mpTitan->GetDebugLabel() );
  mpPauseMenu->SetCheck( TI_CMD_BLOOM,          mpTitan->GetBloom() );
  //mpPauseMenu->SetCheck( TI_CMD_WATERONSPU,     mpTitan->GetEnableSPUWater() );
  mpPauseMenu->SetCheck( TI_CMD_MULTI_PASS,     mpTitan->GetMultiPass() );
  mpPauseMenu->SetCheck( TI_CMD_DISPLAY_STATS,  mpTitanUI->GetDisplayStats() );
  mpPauseMenu->SetCheck( TI_CMD_TOGGLE_SHADOW,  mpTitanUI->GetScene()->AreShadowsEnabled() ? true : false );
  mpPauseMenu->SetCheck( TI_CMD_TOGGLE_PORTALS, mpTitanUI->GetScene()->IsPortalDisplayEnabled() ? true : false );
  mpPauseMenu->SetCheck( TI_CMD_TOGGLE_FLORA,   mpTitanUI->GetScene()->GetRadialFloraMode() ? true : false );
  mpPauseMenu->SetCheck( TI_CMD_TOGGLE_PARTICLES, mpTitanUI->GetScene()->IsParticleSystemEnabled() ? true : false );
  mpPauseMenu->SetCheck( TI_CMD_NORMAL,         mpTitan->GetDrawMode() == TitanI::DrawModeNormal );
  mpPauseMenu->SetCheck( TI_CMD_WIRE_FRAME,     mpTitan->GetDrawMode() == TitanI::DrawModeWire );
  mpPauseMenu->SetCheck( TI_CMD_SOLID_FILL,     mpTitan->GetDrawMode() == TitanI::DrawModeSolid );

  // open window
  mpPauseMenu->Show( true );
  mpPauseMenu->SetFocus();
}

//***********************************************************************
// SetPlayerCount
//***********************************************************************

void TiGameWindow::SetPlayerCount( int32 Count )
{
  mPlayerCount = Count;
  mpCharacterData->SetPlayerCount( mPlayerCount );
  mpActivateWindow->SetPlayerCount( mPlayerCount );
}

//***********************************************************************
// SetHealth
//***********************************************************************

void TiGameWindow::SetHealth( int32 Player, int32 Health )
{
  if( Player >= 0 && Player < PLAYER_MAX )
  {
    int health = SY_CLAMP( Health, 0, mPlayerStats[Player].mHealthMax );
    mPlayerStats[Player].mHealth = health;
  }
}

//***********************************************************************
// SetHealthMax
//***********************************************************************

void TiGameWindow::SetHealthMax( int32 Player, int32 HealthMax )
{
  if( Player >= 0 && Player < PLAYER_MAX )
  {
    mPlayerStats[Player].mHealthMax = SY_MAX( 1, HealthMax );
  }
}

//***********************************************************************
// SetMana
//***********************************************************************

void TiGameWindow::SetMana( int32 Player, int32 Mana )
{
  if( Player >= 0 && Player < PLAYER_MAX )
  {
    int mana = SY_CLAMP( Mana, 0, mPlayerStats[Player].mManaMax );
    mPlayerStats[Player].mMana = mana;
  }
}

//***********************************************************************
// SetManaMax
//***********************************************************************

void TiGameWindow::SetManaMax( int32 Player, int32 ManaMax )
{
  if( Player >= 0 && Player < PLAYER_MAX )
  {
    mPlayerStats[Player].mManaMax = SY_MAX( 1, ManaMax );
  }
}

//***********************************************************************
// SetExperience
//***********************************************************************

void TiGameWindow::SetExperience( int32 Player, int32 Experience )
{
  if( Player >= 0 && Player < PLAYER_MAX )
  {
    int xp = SY_CLAMP( Experience, 0, mPlayerStats[Player].mXPMax );
    mPlayerStats[Player].mXP = xp;
  }
}

//***********************************************************************
// SetExperienceMax
//***********************************************************************

void TiGameWindow::SetExperienceMax( int32 Player, int32 ExperienceMax )
{
  if( Player >= 0 && Player < PLAYER_MAX )
  {
    mPlayerStats[Player].mXPMax = SY_MAX( 1, ExperienceMax );
  }
}

//***********************************************************************
// SetActivateText
//***********************************************************************

void TiGameWindow::SetActivateText( int32 Player, const char8* pText )
{
  mpActivateWindow->SetPlayerText( Player, pText );
}

//***********************************************************************
// SetSubtitleText
//***********************************************************************

void TiGameWindow::SetSubtitleText( const char8* pText )
{
  mpActivateWindow->SetSubtitleText( pText );
}

//***********************************************************************
// SetMinimap
//***********************************************************************

void TiGameWindow::SetMinimap(const SyString&  textureName,
                              const SyVect2&   worldPos1,
                              const SyVect2&   worldPos2,
                              const SyVect2&   mapPos1,
                              const SyVect2&   mapPos2,
                              SyRaster*        raster,
                              SyDictionary*    dictionary)
{
  char buf[512];

  // Pre-load the minimap ESF
  if (strcmp(textureName.AsChar(), "") != 0)
  {
    sprintf(buf, "game_assets/art/ui/%s", textureName.AsChar());
    if (mpTitanUI->GetESFParser().Parse(buf, *mpTitanUI->GetScene()) < 0)
    {
      SyAssert(0);
    }
  }
  else
  {
    buf[0] = 0;
  }

  // Set the minimap texture name
  SyPathname mappath(buf);
  m_MiniMap.Init(mappath.Base(),
                 worldPos1,
                 worldPos2,
                 mapPos1,
                 mapPos2,
                 raster,
                 dictionary);
}

//***********************************************************************
// TogglePawnOutput
//***********************************************************************

void TiGameWindow::TogglePawnOutput()
{
  mpPawnOutput->Show( !mpPawnOutput->IsVisible() );
}

//***********************************************************************
// PawnDebugOutput
//***********************************************************************

void TiGameWindow::PawnDebugOutput( const char8* pText )
{
  mpPawnOutput->Output( pText );
}

//***********************************************************************
// PawnDebugUI
//***********************************************************************

int TiGameWindow::PawnDebugUI( const char8* pSource, int32 LineNum, const char8* pVariables )
{
  mpTitan->PauseGame();
  mpPawnDebug->SetSource( pSource, LineNum );
  mpPawnDebug->SetVariables( pVariables );
  int result = mpPawnDebug->ShowModal();
  mpTitan->UnpauseGame();
  return result;
}

//-----------------------------------------------------------------------
void TiGameWindow::UpdateCharacterData()
{
	if( mpCharacterData != NULL )
	{
		mpCharacterData->Update();
	}
}
