/********************************************************************

Module: TiGlowyText

Author: Jon Wiesman

Description:

Draws super cool glowy text!!1!

Copyright 2006 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiGlowyText.h"
#include "TiUI.h"
#include "math.h"

TiUiParticle::TiUiParticle()
{
  m_bActive = false;
  m_crColorStart = c_crWhite;
  m_crColorEnd = SyColor32F(0.f, 0.f, 0.f, 0.f);
  m_fAge = 0.f;
  m_fMaxAge = 0.f;
  m_pOwner = NULL;
}

void TiUiParticle::Create(SyColor32F crStart, SyColor32F crEnd, TiFPoint fptOrigin, 
                          TiFPoint fptVelocity, float32 fMaxAge, TiUiParticleEmitter *pOwner
                          )
{
  m_crColorStart = crStart;
  m_crColorEnd = crEnd;
  m_fptLocation = fptOrigin;
  m_fptVelocity = fptVelocity;
  m_fAge = 0.f;
  m_fMaxAge = fMaxAge;
  m_pOwner = pOwner;
  m_bActive = true;
}

void TiUiParticle::Kill()
{
  m_bActive = false;
}

SyColor32F TiUiParticle::GetColor() const
{
  SyAssert(m_fMaxAge > 0.f);
  float32 fAgeScalar = m_fAge / m_fMaxAge;

  return SyColor32F(
    m_crColorStart.R + (m_crColorEnd.R - m_crColorStart.R) * fAgeScalar,
    m_crColorStart.G + (m_crColorEnd.G - m_crColorStart.G) * fAgeScalar,
    m_crColorStart.B + (m_crColorEnd.B - m_crColorStart.B) * fAgeScalar,
    m_crColorStart.A + (m_crColorEnd.A - m_crColorStart.A) * fAgeScalar
    );
}

void TiUiParticle::Draw(TiRect rectClip /* = c_rectPS3Screen */) const
{
  if(!m_bActive || m_fAge >= m_fMaxAge)
    return;

  TiPoint pt = m_fptLocation;
  if(!rectClip.PtInRect(pt))
    return;

  TiRect rect(pt, TiSize(1, 1));

  TiUI::Instance()->FillRectangle(rect, GetColor());
}

void TiUiParticle::Update(float32 fElapsed)
{
  if(!m_bActive)
    return;

  SyAssert(m_pOwner != NULL);

  m_fptLocation += (m_fptVelocity * fElapsed);
  m_fptVelocity.y += m_pOwner->GetGravity() * fElapsed;

  m_fAge += fElapsed;
  if(m_fAge >= m_fMaxAge)
    Kill();
}

///////////////////////////////////////////////////////////////////////////////
// TiUiParticleEmitter
///////////////////////////////////////////////////////////////////////////////

TiUiParticleEmitter::TiUiParticleEmitter()
{
  m_fptLocation.Set(0.f, 0.f);
  m_fptVelocity.Set(0.f, 0.f);
  m_fGravity = 150.f;
  m_fHeading = 0.f;
  m_fCone = SY_PI / 4.f;
  m_fEmitRate = 50.f;
  m_fMinEmitSpeed = 125.f;
  m_fMaxEmitSpeed = 175.f;
  m_fMaxAge = 5.f;
  m_fLastEmitElapse = 0.f;
  m_crColorStart = c_crWhite;
  m_crColorEnd = SyColor32F(0.f, 0.f, 0.f, 0.f);
  m_rand.Seed(0);
  m_bEmitting = true;
}

void TiUiParticleEmitter::KillAllParticles()
{
  for(int i = 0; i < kMaxParticles; i++)
  {
    m_arParticles[i].Kill();
  }
}

bool TiUiParticleEmitter::Emit()
{
  if(!m_bEmitting)
    return true;
  TiUiParticle *pParticle = FindAvailableParticle();
  if(pParticle == NULL)
    return false;

  TiFPoint fptVelocity = m_fptVelocity;
  float32 fHeading = m_fHeading - (m_fCone / 2.f) + m_rand.fRand() * m_fCone;

  float32 fSpeed = m_fMinEmitSpeed + (m_fMaxEmitSpeed - m_fMinEmitSpeed) * m_rand.fRand();

  fptVelocity.x += cosf(fHeading) * fSpeed;
  fptVelocity.y += sinf(fHeading) * fSpeed;


  pParticle->Create(m_crColorStart, m_crColorEnd, m_fptLocation, fptVelocity, m_fMaxAge, this);

  return true;
}

void TiUiParticleEmitter::Update(float32 fElapsed)
{
  for(int i = 0; i < kMaxParticles; i++)
  {
    m_arParticles[i].Update(fElapsed);
  }

  if(m_fEmitRate > 0.f)
  {
    float32 fEmitDelay = 1.f / m_fEmitRate;
    m_fLastEmitElapse += fElapsed;
    while(m_fLastEmitElapse > fEmitDelay)
    {
      if(Emit())
        m_fLastEmitElapse -= fEmitDelay;
      else
        break;
    }
  }
}

void TiUiParticleEmitter::Draw(TiRect rectClip) const
{
  for(int i = 0; i < kMaxParticles; i++)
  {
    m_arParticles[i].Draw(rectClip);
  }
}

TiUiParticle *TiUiParticleEmitter::FindAvailableParticle()
{
  for(int i = 0; i < kMaxParticles; i++)
  {
    if(!m_arParticles[i].IsActive())
      return &m_arParticles[i];
  }
  return NULL;
}


