/********************************************************************

  Module: TiKeyboard

  Author: Carlo Ordonez

  Description:

    Virtual keyboard

  Copyright 2006 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiKeyboard.h"
#include "TiUIControl.h"
#include "TiFocusMessage.h"
#include "TiUI.h"

static const char8 sKeyMap[TiKeyboard::sRows][TiKeyboard::sColumns] =
{
  { 'A', 'B', 'C', 'D', 'E', 'F' },
  { 'G', 'H', 'I', 'J', 'K', 'L' },
  { 'M', 'N', 'O', 'P', 'Q', 'R' },
  { 'S', 'T', 'U', 'V', 'W', 'X' },
  { 'Y', 'Z', '0', '1', '2', '3' },
  { '4', '5', '6', '7', '8', '9' }
};

//***********************************************************************
// Constructor/Destructor
//***********************************************************************

TiKeyboard::TiKeyboard( TiWindow* pParent )
: TiWindow( pParent )
{
  mPlayerId = 0;
  mpCtrl = SyNew TiUIControl();
  mText[0] = 0;
  mRow = 0;
  mCol = 0;
  mMaxLen = sTextMax;
  mShiftDown = false;
}

TiKeyboard::~TiKeyboard()
{
  delete mpCtrl;
}

//***********************************************************************
// SetTitle
//***********************************************************************

void TiKeyboard::SetTitle( const char8* pTitle )
{
  mTitle.Init( pTitle );
}

//***********************************************************************
// GetTitle
//***********************************************************************

const char8* TiKeyboard::GetTitle() const
{
  return( mTitle.AsChar() );
}

//***********************************************************************
// SetMaxLen
//***********************************************************************

void TiKeyboard::SetMaxLen( int32 MaxLen )
{
  mMaxLen = SY_MIN( MaxLen, sTextMax );
}

//***********************************************************************
// GetMaxLen
//***********************************************************************

int32 TiKeyboard::GetMaxLen() const
{
  return( mMaxLen );
}

//***********************************************************************
// SetText
//***********************************************************************

void TiKeyboard::SetText( const char8* pText )
{
  memset( mText, 0, sizeof(mText) );
  SyStr::strncpy( mText, pText, sTextMax-1 );
}

//***********************************************************************
// GetText
//***********************************************************************

const char8* TiKeyboard::GetText() const
{
  return( &mText[0] );
}

//***********************************************************************
// DrawSelf
//***********************************************************************

void TiKeyboard::DrawSelf()
{
  int32 margin = 20;
  int32 tbWidth = mpCtrl->GetTextWidth( "|sBackspace  |aDone" );
  int32 width = tbWidth + (margin<<1);
  int32 rows = sRows + 9;
  mpCtrl->Init( width, rows );
  mpCtrl->Draw();

  // title
  mpCtrl->DrawTextBox( mTitle, margin, 0, tbWidth, 1, TiUIControl::kNormal, TI_TF_SHADOW|TI_TF_HALIGN_CENTER );

  // text
  mpCtrl->DrawTextBox( mText, margin, 2, tbWidth, 1, TiUIControl::kNormal, TI_TF_SHADOW );

  // caret
  SyVect2I rect[2];
  rect[0] = mpCtrl->GetClientUL();
  rect[0].X += margin + TiUI::Instance()->GetStringWidth(mText);
  rect[0].Y += (TiUI::Instance()->GetFontHeight() * 3);
  rect[1] = rect[0] + SyVect2I( TiUI::Instance()->GetStringWidth("n"), 3 );
  TiUI::Instance()->FillRectangle( rect[0], rect[1], SyColor32F(1,1,1,mpCtrl->GetTickPulse()) );

  // keys
  int32 incr = tbWidth / sColumns;
  for( int32 row = 0; row < sRows; row++ )
  {
    int32 x = margin;
    for( int32 col = 0; col < sColumns; col++ )
    {
      char8 c[2];
      c[0] = sKeyMap[row][col];
      c[1] = 0;

      if( c[0] >= 'A' && c[0] <= 'Z' && !mShiftDown )
      {
        c[0] = c[0] - 'A' + 'a';
      }

      bool selected = mRow == row && mCol == col;
      TiUIControl::TextColor color = selected ? TiUIControl::kSelected : TiUIControl::kNormal;

      mpCtrl->DrawTextBox( c, x, row+4, incr, 1, color, TI_TF_SHADOW|TI_TF_HALIGN_CENTER );
      x += incr;
    }
  }

  // buttons
  mpCtrl->DrawTextBox( "|xAccept  |rShift", margin, rows-4, tbWidth, 1, TiUIControl::kNormal, TI_TF_SHADOW|TI_TF_HALIGN_CENTER );
  mpCtrl->DrawTextBox( "|sBackspace  |aDone", margin, rows-2, tbWidth, 1, TiUIControl::kNormal, TI_TF_SHADOW|TI_TF_HALIGN_CENTER );
}

//***********************************************************************
// ProcessButtonDown
//***********************************************************************

bool TiKeyboard::ProcessButtonDown( int32 DeviceId, int32 ButtonId )
{
  if( DeviceId != mPlayerId ) { return( true ); }

  switch( TiWindow::GetButton( ButtonId ) )
  {
  case TiButtonMap::kCircle:
    CloseWindow();
    return( true );

  case TiButtonMap::kUp:
    Move( 0, -1 );
    return( true );

  case TiButtonMap::kDown:
    Move( 0, 1 ); 
    return( true );

  case TiButtonMap::kLeft:
    Move( -1, 0 );
    return( true );

  case TiButtonMap::kRight:
    Move( 1, 0 );
    return( true );

  case TiButtonMap::kX:
    Select();
    return( true );

  case TiButtonMap::kSquare:
    Backspace();
    return( true );

  case TiButtonMap::kR1:
    mShiftDown = true;
    return( true );

  case TiButtonMap::kStart:
    Accept();
    return( true );

  default:
    return( false );
  }
}

//***********************************************************************
// ProcessButtonUp
//***********************************************************************

bool TiKeyboard::ProcessButtonUp( int32 DeviceId, int32 ButtonId )
{
  if( DeviceId != mPlayerId ) { return( true ); }

  switch( TiWindow::GetButton( ButtonId ) )
  {
  case TiButtonMap::kR1:
    mShiftDown = false;
    return( true );

  default:
    return( false );
  }
}

//***********************************************************************
// ProcessMessage
//***********************************************************************

void TiKeyboard::ProcessMessage( TiWindowMessage* pMessage )
{
  switch( pMessage->GetMessageId() )
  {
  case TIWINDOW_SETFOCUS:
    {
      TiFocusMessage* pMsg = static_cast<TiFocusMessage*>(pMessage);
      TiWindow* pOld = pMsg->GetOldWindow();
      TiWindow* pNew = pMsg->GetNewWindow();

      if( pNew == this && pOld != this )
      {
        // just gained focus
        mRow = 0;
        mCol = 0;
        mText[0] = 0;

        mpCtrl->Init( 300, 8 );
        mpCtrl->Close( true );
        mpCtrl->Open( false );
      }
    }
    break;

  default:
    TiWindow::ProcessMessage( pMessage );
    break;
  }
}

//***********************************************************************
// Move
//***********************************************************************

void TiKeyboard::Move( int32 X, int32 Y )
{
  TiUI::Instance()->PlaySound( TiUI::kSoundTick );
  mRow += Y;
  if( mRow >= sRows )
  {
    mRow = 0;
  }
  else if( mRow < 0 )
  {
    mRow = sRows - 1;
  }

  mCol += X;
  if( mCol >= sColumns )
  {
    mCol = 0;
  }
  else if( mCol < 0 )
  {
    mCol = sColumns - 1;
  }
}

//***********************************************************************
// Select
//***********************************************************************

void TiKeyboard::Select()
{
  TiUI::Instance()->PlaySound( TiUI::kSoundName );

  if( SyStr::strlen( mText ) < mMaxLen-1 )
  {
    char8 c[2];
    c[0] = sKeyMap[mRow][mCol];
    c[1] = 0;

    if( c[0] >= 'A' && c[0] <= 'Z' && !mShiftDown )
    {
      c[0] = c[0] - 'A' + 'a';
    }

    strcat( mText, c );
  }
}

//***********************************************************************
// Backspace
//***********************************************************************

void TiKeyboard::Backspace()
{
  TiUI::Instance()->PlaySound( TiUI::kSoundName );

  int32 len = SyStr::strlen( mText );
  if( len > 0 )
  {
    mText[len-1] = 0;
  }
}

//***********************************************************************
// Accept
//***********************************************************************

void TiKeyboard::Accept()
{
  TiUI::Instance()->PlaySound( TiUI::kSoundSelect );

  TiWindowMessage msg( this, TIWINDOW_KB_ACCEPT );
  ProcessMessage( &msg );
}

//***********************************************************************
// IsVisible
//***********************************************************************

bool TiKeyboard::IsVisible() const
{
  return( mVisible || mpCtrl->IsOpen() || mpCtrl->IsOpening() || mpCtrl->IsClosing() );
}

//***********************************************************************
// Show
//***********************************************************************

void TiKeyboard::Show( bool Visible ) 
{
  mVisible = Visible;
  if( Visible )
  {
    mpCtrl->Open();
  }
  else
  {
    mpCtrl->Close( true );
  }
}
