/********************************************************************

Module: TiLoadSaveGameWindow

Author: Christian Ebbecke

Description:

Temp window for loading/saving of games

Copyright 2006 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiLoadSaveGameWindow.h"

#include "TitanUI.h"
#include "TiFocusMessage.h"
#include "TiFont.h"
#include "TiStartMessage.h"
#include "TiUI.h"
#include "TiUIControl.h"


//-----------------------------------------------------------------------
//-----------------------------------------------------------------------
namespace
{
	static TiRect c_rectOuterFrame(TiPoint(400, 140), TiSize(410, 400));
	static TiRect c_rectTitlebar(TiPoint(16, 7), TiSize(316, 27));
	const TiRect c_rectEquipmentList(TiPoint(0, 40), TiSize(340, 334));
	const int32 c_nVisItemsInList = 5;
	const int32 c_nListItemHeight = 56;
};
//-----------------------------------------------------------------------
//-----------------------------------------------------------------------



//***********************************************************************
// Constructor/Destructor
//***********************************************************************
TiLoadSaveGameWindow::TiLoadSaveGameWindow( TiWindow* pParent )
	: TiWindow( pParent )
{
	mpCtrl = SyNew TiUIControl();

	m_cfd.SetScrollbar(true);
	m_bdSelection.Initialize("Selection");

	mSelection = 0;
	mForLoading = 0;

	m_rect = c_rectOuterFrame;
}

TiLoadSaveGameWindow::~TiLoadSaveGameWindow()
{
	delete mpCtrl;
}

//***********************************************************************
// Init
//***********************************************************************
void TiLoadSaveGameWindow::Init( bool bForLoading )
{
	if( bForLoading )
	{
		mForLoading = 1;
	}
	else
	{
		mForLoading = 0;
	}

	mSelection = 0;
	m_cfd.SetPos(0);
	m_cfd.SetRange( TitanUI::Instance()->GetTitanI()->SaveGameGetCount() );
}

//***********************************************************************
// DrawSelf
//***********************************************************************
void TiLoadSaveGameWindow::DrawSelf()
{
	TiFont font(TiUI::Instance()->GetFontHandle("Default"));

	int32 nRows = 17;
	int32 nMargin = 20;
	int32 nWidth = m_rect.Width() - (nMargin<<1);

	mpCtrl->Init( nWidth, nRows );
	mpCtrl->Draw();

	TiPoint pt;
	pt.x = mpCtrl->GetClientUL().X;
	pt.y = mpCtrl->GetClientUL().Y;

	char8 szBuffer[256] = "";
	static SyResourceID idLoadGame = SyHashResourceID("UiMainMenuLoadGame");
	static SyResourceID idSaveGame = SyHashResourceID("ACTION_SAVEGAME");

	if( mForLoading )
	{
		TitanUI::Instance()->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), idLoadGame);
	}
	else
	{
		TitanUI::Instance()->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), idSaveGame);
	}
	mpCtrl->DrawTextBox( szBuffer, 0, 0, nWidth, 1, TiUIControl::kNormal, TI_TF_HALIGN_CENTER|TI_TF_SHADOW );


	TiRect rectSelect = c_rectEquipmentList;
	pt.x += (( nWidth - c_rectEquipmentList.Width() ) >> 1 );
	rectSelect += pt;
	m_cfd.Draw(rectSelect);

	TiRect rectList = m_cfd.GetContentRect(rectSelect);
	TiRect rectItem = rectList;
	rectItem.bottom = rectItem.top + c_nListItemHeight;

	// draw game list
	for( int32 i = m_cfd.GetPos(); (( i < m_cfd.GetRange() ) && ( i < m_cfd.GetPos() + c_nVisItemsInList )); ++i )
	{
		SyColor32F crText = c_crWhite;
		if( i == mSelection )
		{
			m_bdSelection.Draw( rectItem );
			crText = c_crItemEquipped;
		}

		TiRect rectText = rectItem;
		rectText.DeflateRect( 8, 0 );

		TitanSave pInfo;
		TitanUI::Instance()->GetTitanI()->SaveGameGetInfo( i, &pInfo );
		font.Draw( pInfo.mName, rectText, crText, TI_TF_VCENTER | TI_TF_SHADOW );

		rectItem += TiPoint(0, c_nListItemHeight);
	}

	if( mForLoading )
	{
		mpCtrl->DrawTextBox( "|xLoad  |cCancel", 0, nRows-2, nWidth, 1, TiUIControl::kNormal, TI_TF_HALIGN_CENTER|TI_TF_SHADOW );
	}
	else
	{
		mpCtrl->DrawTextBox( "|xSave  |tNew  |cCancel", 0, nRows-2, nWidth, 1, TiUIControl::kNormal, TI_TF_HALIGN_CENTER|TI_TF_SHADOW );
	}
}

//***********************************************************************
// ProcessButtonDown
//***********************************************************************
bool TiLoadSaveGameWindow::ProcessButtonDown( int32 DeviceId, int32 ButtonId )
{
	switch( TiWindow::GetButton( ButtonId ) )
	{
	case TiButtonMap::kCircle:
		Close();
		return( true );

	case TiButtonMap::kTriangle:
		NewGame();
		return( true );

	case TiButtonMap::kUp:
		TiUI::Instance()->PlaySound( TiUI::kSoundTick );
		PrevSelection();
		return( true );

	case TiButtonMap::kDown:
		TiUI::Instance()->PlaySound( TiUI::kSoundTick );
		NextSelection();
		return( true );

	case TiButtonMap::kX:
		Select();
		return( true );

	default:
		return( TiWindow::ProcessButtonDown( DeviceId, ButtonId ) );
	}
}

//***********************************************************************
// ProcessMessage
//***********************************************************************
void TiLoadSaveGameWindow::ProcessMessage( TiWindowMessage* pMessage )
{
	switch( pMessage->GetMessageId() )
	{
	case TIWINDOW_SETFOCUS:
		// just gained focus
		mpCtrl->Close( true );
		mpCtrl->Open();
		break;

	default:
		TiWindow::ProcessMessage( pMessage );
		break;
	}
}

//***********************************************************************
// Select
//***********************************************************************
void TiLoadSaveGameWindow::Select()
{
	if( m_cfd.GetRange() > 0 )
	{
		if( mForLoading )
		{
			TiStartMessage msg( this, TI_START_LOAD_GAME );
			msg.SetLoadGame( mSelection );
			TitanUI::Instance()->ProcessCommandMessage( &msg );
		}
		else
		{
			SyString desc( TitanUI::Instance()->GetTitanI()->GetLevelName() );
			TitanUI::Instance()->GetTitanI()->SaveGameOverwrite( mSelection, desc );
		}
	}
	Close();
}

//***********************************************************************
// New Game
//***********************************************************************
void TiLoadSaveGameWindow::NewGame()
{
	if( !mForLoading )
	{
		SyString desc( TitanUI::Instance()->GetTitanI()->GetLevelName() );
		TitanUI::Instance()->GetTitanI()->SaveGameCreate( desc );

		Close();
	}
}

//***********************************************************************
// Next Selection
//***********************************************************************
void TiLoadSaveGameWindow::NextSelection()
{
	if( mSelection < m_cfd.GetRange()-1 )
	{
		++mSelection;
	}
}

//***********************************************************************
// Previous Selection
//***********************************************************************
void TiLoadSaveGameWindow::PrevSelection()
{
	if( mSelection > 0 )
	{
		--mSelection;
	}
}

//***********************************************************************
// Close
//***********************************************************************
void TiLoadSaveGameWindow::Close()
{
	TiUI::Instance()->PlaySound( TiUI::kSoundBack );
	CloseWindow();
}
