/********************************************************************

  Module: TiMenu

  Author: Carlo Ordonez

  Description:

    Game menu classes

  Copyright 2006 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiMenu.h"
#include "SyStr.h"

//***********************************************************************
// TiMenu class
//***********************************************************************

TiMenu::TiMenu()
{
  mpParent = NULL;
  mSelection = 0;
  mScroll = 0;
  mWrapSelections = 0;
}

TiMenu::~TiMenu()
{
  mMenuItems.clear();
}

int32 TiMenu::Init( TiMenu* pParent, const char8* pTitle )
{
  mpParent = pParent;
  mTitle = pTitle;
  mSelection = 0;
  mScroll = 0;
  mMenuItems.clear();
  mWrapSelections = 0;
  return( 0 );
}

void TiMenu::AddItem( int32 Id, const char8* pText, TiMenuItem::Type Type, TiMenu* pSubmenu )
{
  TiMenuItem newItem( this, Id, pText, Type, pSubmenu );
  mMenuItems.push_back( newItem );

  // by default, wrap any list that has > 2 items in it
  if( mMenuItems.size() > 2 )
  {
	  mWrapSelections = 1;
  }
}

bool TiMenu::GetCheck( int32 MenuItemId ) const
{
  for( uint32 i = 0; i < mMenuItems.size(); i++ )
  {
    if( mMenuItems[i].mId == MenuItemId )
    {
      return( mMenuItems[i].GetCheck() );
    }
  }

  // not found
  return( false );
}

void TiMenu::SetCheck( int32 MenuItemId, bool Check )
{
  for( uint32 i = 0; i < mMenuItems.size(); i++ )
  {
    TiMenuItem& item = mMenuItems[i];
    if( item.mId == MenuItemId )
    {
      if( item.mType == TiMenuItem::kCheck )
      {
        // set it straightaway
        item.SetCheck( Check );
      }
      else if( item.mType == TiMenuItem::kRadio )
      {
        if( Check ) // cannot de-select a radio item
        {
          // de-select the others
          for( uint32 j = 0; j < mMenuItems.size(); j++ )
          {
            if( mMenuItems[j].mId != item.mId )
            {
              mMenuItems[j].SetCheck( false );
            }
          }

          // select this one
          item.SetCheck( true );
        }
      }

      return;
    }
  }
}

void TiMenu::SetItemText( int32 MenuItemId, const char8 *pText )
{
  for( uint32 i = 0; i < mMenuItems.size(); i++ )
  {
    TiMenuItem& item = mMenuItems[i];
    if( item.mId == MenuItemId )
    {
      SyStr::strncpy(item.mText, pText, sizeof(item.mText) );
      return;
    }
  }
}

void TiMenu::Next()
{
	if( mMenuItems.size() < 2 )
	{
		return;
	}

	int next = mSelection + 1;
	int nLoops = 0;	// to prevent infinite looping....
	while( nLoops < 2 )
	{
		if( next >= (int)mMenuItems.size() )
		{
			if( mWrapSelections )
			{
				next = 0;
				++nLoops;
			}
			else
			{
				return;
			}
		}

		if( mMenuItems[next].mType != TiMenuItem::kSeperator )
		{
			mSelection = next;
			return;
		}

		++next;
	}
}

void TiMenu::Prev()
{
	if( mMenuItems.size() < 2 )
	{
		return;
	}

	int prev = mSelection - 1;
	int nLoops = 0;	// to prevent infinite looping....
	while( nLoops < 2 )
	{
		if( prev < 0 )
		{
			if( mWrapSelections )
			{
				prev = ( (int32)mMenuItems.size() - 1 );
				++nLoops;
			}
			else
			{
				return;
			}
		}

		if( mMenuItems[prev].mType != TiMenuItem::kSeperator )
		{
			mSelection = prev;
			return;
		}

		--prev;
	}
}

int32 TiMenu::Find( int32 MenuItemId )
{
  for( uint32 i = 0; i < mMenuItems.size(); i++ )
  {
    if( mMenuItems[i].mId == MenuItemId )
    {
      return( i );
    }
  }

  // not found
  return( -1 );
}

//***********************************************************************
// TiMenuItem class
//***********************************************************************

TiMenuItem::TiMenuItem( TiMenu* pParent, int32 Id, const char8* pText, Type type, TiMenu* pSubmenu )
{
  mpParent  = pParent;
  mId       = Id;
  mType     = type;
  mpSubmenu = pSubmenu;
  mChecked  = false;

  SyStr::strncpy( mText, pText, sizeof(mText) );
}

TiMenuItem::~TiMenuItem()
{
}
