/********************************************************************

  Module: TiSceneController

  Author: Carlo Ordonez

  Description:

    Scene control for front end

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiSceneController.h"
#include "TitanI.h"
#include "SyESFParse.h"
#include "SyCamera.h"
#include "SyDebug.h"

static const SyVect3 sStart(0,1,-24);

//***********************************************************************
// Constructor/Destructor
//***********************************************************************

TiSceneController::TiSceneController()
{
  mpTitanUI = NULL;
  mpCamera = NULL;
  mSceneLoaded = false;
  for( int i = 0; i < 3; i++ )
  {
    mhActors[i] = SyActorNull;
  }
}

TiSceneController::~TiSceneController()
{
  delete mpCamera;
}

//***********************************************************************
// Init
//***********************************************************************

int TiSceneController::Init( TitanUII* pUI, float32 ViewAngle, float32 AspectRatio )
{
  mpTitanUI = pUI;
  mViewAngle = ViewAngle;
  mAspectRatio = AspectRatio;

  if( mpCamera == NULL )
  {
    mpCamera = SyNew SyCamera();
  }

  mpCamera->Init( *mpTitanUI->GetScene()->GetRasterDev(),
                  SyVect3(0,0,0),
                  SyVect3(0,0,0), 
                  mViewAngle,
                  mAspectRatio, 
                  .5f, 
                  10000.f );

  return( 0 );
}

//***********************************************************************
// Reset
//***********************************************************************

int TiSceneController::Reset()
{
#if 0       // not needed here? - JEFFP
  mpTitanUI->SceneClearAndInit();
#endif

  mSceneLoaded = false;
  for( int i = 0; i < 3; i++ )
  {
    mhActors[i] = SyActorNull;
  }
  return( 0 );
}

//***********************************************************************
// LoadScene
//***********************************************************************

int TiSceneController::LoadScene( const char* pFilename )
{
  Reset();

  SyScene* pScene = mpTitanUI->GetScene();
  SyESFParse& parser = mpTitanUI->GetESFParser();
  if( parser.ParseWorld( pFilename, 0, *pScene ) < 0 )
  {
    return( -1 );
  }

  int sprite = parser.ParseSprite( "game_assets\\art\\characters\\brute.esf", *pScene );
  if( sprite < 0 )
  {
    return( -1 );
  }

  for( int i = 0; i < 3; i++ )
  {
    mhActors[i] = pScene->CreateActor( SyVect3(float32(i*-6),0,0), SyVect3(SY_PI,0,0), 1.f, sprite );
    if( mhActors[i] == SyActorNull )
    {
      return( -1 );
    }
    SyCSprite* pSprite = static_cast<SyCSprite*>(pScene->GetActorSpritePtr(mhActors[i]));
    if( pSprite->HasAnim(1) )
    {
      pSprite->AppendAnimPlay( 1/*idle*/, 0, *pScene );
    }
  }

  mpCamera->Init( *mpTitanUI->GetScene()->GetRasterDev(),
                  sStart, 
                  SyVect3(0,0,1),
                  mpCamera->GetViewAngle(),
                  mpCamera->GetAspectRatio(),
                  mpCamera->GetNear(),
                  mpCamera->GetFar() );

  mSceneLoaded = true;
  return( 0 );
}

//***********************************************************************
// Render
//***********************************************************************

void TiSceneController::Render()
{
  if( mSceneLoaded )
  {
    SyScene* pScene = mpTitanUI->GetScene();
    //pScene->SetTime( SyTimer::GetTime() );
    pScene->Update( *mpCamera );
    pScene->RenderHUD();
  }
}

//***********************************************************************
// GotoActor
//***********************************************************************

void TiSceneController::GotoActor( int Index )
{
  SyScene* pScene = mpTitanUI->GetScene();
  SyVect3 loc = sStart;

  if( Index >= 0 && mhActors[Index] != SyActorNull )
  {
    loc = pScene->GetActorLocation( mhActors[Index] );
    loc.X += 2;
    loc.Y += 1;
    loc.Z -= 5;
  }

  mpCamera->Init( *pScene->GetRasterDev(),
                  loc,
                  SyVect3(0,0,1),
                  mpCamera->GetViewAngle(),
                  mpCamera->GetAspectRatio(),
                  mpCamera->GetNear(),
                  mpCamera->GetFar() );
}
