/********************************************************************

  Module: TiStartWindow

  Author: Carlo Ordonez
          JonW
          Bijan Forutanpour

  Description:

    Front screen

  Copyright 2006 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiStartWindow.h"
#include "TiClassWindow.h"
#include "TiNameWindow.h"
#include "TiUIControl.h"
#include "TiFocusMessage.h"
#include "TiCommandMessage.h"
#include "TiStartMessage.h"
#include "TiCharacterCreateMessage.h"
#include "TitanI.h"
#include "TiUI.h"
#include "InputConsole.h"
#include "SyScene.h"
#include "TiFont.h"
#include "T4utf.h"
#include "SyESFParse.h"

#include "TiLoadSaveGameWindow.h"


//***********************************************************************
// Constructor/Destructor
//***********************************************************************
const TiSize c_sizeSelection(150, 37);

TiStartWindow::TiStartWindow( TiApp* pApp, TitanUII *pTitanUI )
  : TiTopLevelWindow( pApp ), mpTitanUI(pTitanUI)
{
  mTime = TiUI::Instance()->GetTime();
  mpCtrl = SyNew TiUIControl();
  mMode = kEsrbLogo;
  mpCurrentMenu = &mMenuMain;
  mNumPlayers = 1;
  mInternet = false;
  mCampaign = true;
  mClass = 0;
  mfTargetWidth = 200.f;
  mfCurrentWidth = 200.f;
  mfWidthVelocity = 0.f;
  TiMenuItem::Type type = TiMenuItem::kNormal;

  /*
   * Get the commonly used Titan, TitanUI, Scene, and Raster pointers
   */
  mpScene =   mpTitanUI->GetScene();
  mpRaster =  mpScene->GetRasterDev();

  // the text on these menu options is unimportant, since I immediately
  // call a function to get the text from T4

  // populate menus
  mMenuMain.Init( NULL, "" );
  mMenuMain.AddItem( Id_Go,       "START GAME", type, NULL );
  //mMenuMain.AddItem( Id_NewGame,  "Start New Game", type, &mMenuNewGame );
  mMenuMain.AddItem( Id_LoadGame, "Load Game", type, NULL );
  mMenuMain.AddItem( Id_Options,  "Options", type, &mMenuOptions );
  mMenuMain.AddItem( Id_Downloads, "Downloads", type, NULL);
  mMenuMain.AddItem( Id_Credits,  "Credits", type, NULL );
#ifdef WIN32
  mMenuMain.AddItem( Id_Exit_Titan,  "Exit Titan", type, NULL );
#endif

  mMenuOptions.Init(&mMenuMain, "Options");
  mMenuOptions.AddItem(Id_Language_English,  "English",  TiMenuItem::kCheck, NULL);
  mMenuOptions.AddItem(Id_Language_French,   "French",   TiMenuItem::kCheck, NULL);
  mMenuOptions.AddItem(Id_Language_German,   "German",   TiMenuItem::kCheck, NULL);
  mMenuOptions.SetCheck(Id_Language_English, true);

  mMenuNewGame.Init( &mMenuMain, "New Game" );
  mMenuNewGame.AddItem( Id_Normal,    "Normal Game", type, &mMenuPlayers );
  mMenuNewGame.AddItem( Id_Internet,  "Internet Game", type, NULL );

  mMenuPlayers.Init( &mMenuNewGame, "Number of Players" );
  mMenuPlayers.AddItem( Id_1Player,   "1 Player", type, &mMenuGameType );
  mMenuPlayers.AddItem( Id_2Players,  "2 Players", type, NULL );
  mMenuPlayers.AddItem( Id_3Players,  "3 Players", type, NULL );
  mMenuPlayers.AddItem( Id_4Players,  "4 Players", type, NULL );

  mMenuGameType.Init( &mMenuPlayers, "Game Type" );
  mMenuGameType.AddItem( Id_Campaign, "Campaign", type, &mMenuDifficulty );
  mMenuGameType.AddItem( Id_Arena,    "Arena", type, NULL );

  mMenuDifficulty.Init( &mMenuGameType, "Difficulty" );
  mMenuDifficulty.AddItem( Id_Easy,   "Adventurous", type, NULL );
  mMenuDifficulty.AddItem( Id_Medium, "Courageous", type, NULL );
  mMenuDifficulty.AddItem( Id_Hard,   "Champion", type, NULL );
  mMenuDifficulty.AddItem( Id_Expert, "Legendary", type, NULL );

  // character creation windows
  mpClassWindow = SyNew TiClassWindow( this );
  mpClassWindow->Show( false );
  mpNameWindow = SyNew TiNameWindow( this );
  mpNameWindow->Show( false );
  mpLoadGameWindow = SyNew TiLoadSaveGameWindow( this );
  mpLoadGameWindow->Show( false );

  mSprMenuBackground.SetInfo("title_menu.tga");
  mSprLogo.SetInfo("title_splash.tga");
  mSprLeftNinjaStar.SetInfo("ui_menu_selection_ninjastar_left.tga");
  mSprRightNinjaStar.SetInfo("ui_menu_selection_ninjastar_right.tga");
  mBoxSelection.Initialize("FrontEndSelection");
  mSprChecked.SetInfo("menu_dialog_checkbox_check.tga");
  mSprUnchecked.SetInfo("menu_dialog_checkbox.tga");
  mSprSoeLogo.SetInfo("soe_logo.tga");
  mSprBSDownloads.SetInfo("downloads.tga");

  mEmitterLeft.SetHeading(3.f * SY_PI / 2.f);
  mEmitterRight.SetHeading(3.f * SY_PI / 2.f);
  mEmitterLeft.SetEmitRate(25.f);
  mEmitterRight.SetEmitRate(25.f);
  mEmitterLeft.SetColors(SyColor32F(1.f, 1.f, .5f, 1.f), SyColor32F(0.f, .0f, 0.5f, 0.5f));
  mEmitterRight.SetColors(SyColor32F(1.f, 1.f, .5f, 1.f), SyColor32F(0.f, .0f, 0.5f, 0.5f));
  mEmitterLeft.StopEmitting();
  mEmitterRight.StopEmitting();


  /*
   * Open the Level ESF File we're using as background
   */
  SyESFParse parser;

  char levelName [] = "game_assets/art/levels/palace_1.esf";
  int result = parser.Parse (levelName,  *mpScene);
  if (result == -1)
  {
    SyOutputString("Could not find or open file (%s)", levelName);
    return;
  }

  int playerSprite = parser.ParseSprite ("game_assets/art/characters/brute.esf", *mpScene);
  mPaperDollSprite = mpScene->CopySprite (playerSprite, 0);

  /*
   * Set up camera info
   */
  SyVect3 cameraPosition (-1, 1.25, 0.f);
  SyVect3 cameraLookatDirection( -1.f, 0, 0);
  float32 screenWidth = (float32)mpRaster->GetScreenWidth();
  float32 screenHeight = (float32)mpRaster->GetScreenHeight();
  float32 screenAspect = screenWidth / screenHeight;
  mCamera.Init (*mpRaster, cameraPosition, cameraLookatDirection, 55.f, screenAspect, .1f, 1000.f);
}

TiStartWindow::~TiStartWindow()
{
  delete mpCtrl;
}

void TiStartWindow::OnProcessFrame()
{
  static bool bT4Init = false;
  if(!bT4Init)
  {
    // the aforementioned function load menu text
    LoadMenuText();
    bT4Init = true;
  }
}

//***********************************************************************
// DrawSelf
//***********************************************************************

void TiStartWindow::DrawSelf()
{
  TiSize sizeScreen;
  TiUI::Instance()->GetDisplaySize( &sizeScreen.cx, &sizeScreen.cy );
  TiRect rectScreen(TiPoint(0, 0), sizeScreen);

  if( false && mMode == kEsrbLogo )
  {
    TiUI::Instance()->FillRectangle(rectScreen, c_crBlack);
    TiSize sizeLogo(442, 299);
    TiPoint ptBG((sizeScreen.cx - sizeLogo.cx) / 2, (sizeScreen.cy - sizeLogo.cy) / 2);
    mSprSoeLogo.Stretch(TiRect(ptBG, sizeLogo));
  }
  else if( mMode == kEsrbLogo || mMode == kSoeLogo )
  {
    TiPoint ptBG((sizeScreen.cx - mSprLogo.GetSize().cx) / 2, 0);
    if(sizeScreen.cx > mSprLogo.GetSize().cx || sizeScreen.cy > mSprLogo.GetSize().cy)
    {
      mSprLogo.Stretch(rectScreen); 
    }
    else
    {
      mSprLogo.Draw(ptBG);
    }
  }
  else if(mMode == kBSDownloads)
  {
	  mSprBSDownloads.Stretch(rectScreen); 
  }
  else if(mMode == kLoadGame)
  {
	  TiPoint ptBG((sizeScreen.cx - mSprMenuBackground.GetSize().cx) / 2, 0);

	  if(sizeScreen.cx > mSprMenuBackground.GetSize().cx || sizeScreen.cy > mSprMenuBackground.GetSize().cy)
	  {
		  mSprMenuBackground.Stretch(rectScreen); 
	  }
	  else
	  {
		  mSprMenuBackground.Draw(ptBG);
	  }
  }
  else if( mMode == kStartScreen )
  {
    UpdateMenuSelectionWidth();

    // the background may be larger than the screen so
    // ptBG.x might be negative
    TiPoint ptBG((sizeScreen.cx - mSprMenuBackground.GetSize().cx) / 2, 0);

#if 0
    /*
     * Draw 2D background - soon to be removed
     */
    if(sizeScreen.cx > mSprMenuBackground.GetSize().cx || sizeScreen.cy > mSprMenuBackground.GetSize().cy)
    {
      mSprMenuBackground.Stretch(rectScreen); 
    }
    else
    {
      mSprMenuBackground.Draw(ptBG);
    }
#endif

    /*
     * Draw 3D background - Set up raster window 3D model/view transformations
     */
    SyMatrix44 rasterWorldMatrix;
    SyMatrix44 rasterViewMatrix = mCamera.GetView();
    SyMatrix44 rasterProjectionMatrix = mCamera.GetProjection();
    mpRaster->SetWorld (rasterWorldMatrix);
    mpRaster->SetView (rasterViewMatrix);
    mpRaster->SetProjection (rasterProjectionMatrix);
    mpRaster->EnableAmbient (1);
    int result;
    // result = mpScene->RenderSprite (mCamera, mPaperDollSprite);
    result = mpScene->Render(mCamera);

    /*
     * START 2D menu/text rendering
     */
    mpRaster->Begin2D();

    TiPoint ptFirstItem(ptBG.x + 675, 300); 
    
    TiRect rect(ptFirstItem, c_sizeSelection);
    TiFont font(TiUI::Instance()->GetFontHandle("Default"));
    std::vector< TiMenuItem >& menuItems = mpCurrentMenu->mMenuItems;
    for(int32 i = 0; i < (int32)menuItems.size(); i++)
    {
      SyColor32F crText = c_crMenuText;
      uint32 uTextFlags = TI_TF_HALIGN_CENTER | TI_TF_SHADOW | TI_TF_VCENTER;
      if(i == mpCurrentMenu->mSelection)
      {
        TiRect rectSel = rect;
        int32 nCenterX = rect.CenterTop().x;
        int nWidth = (int)(mfCurrentWidth + 0.5f);
        rectSel.left = nCenterX - nWidth / 2;
        rectSel.right = rectSel.left + nWidth;

        crText = c_crWhite;
        uTextFlags |= TI_TF_GLOW;
        //mBoxSelection.Draw(rectSel);

        TiPoint ptStar = rectSel.LeftCenter();
        mEmitterLeft.SetLocation(ptStar - TiPoint(15, 0));
        mEmitterLeft.SetVelocity(TiFPoint(-mfWidthVelocity, 0.f));
        ptStar.x -= mSprLeftNinjaStar.GetSize().cx;
        ptStar.y -= mSprLeftNinjaStar.GetSize().cy / 2;
        //mSprLeftNinjaStar.Draw(ptStar);

        ptStar = rectSel.RightCenter();
        mEmitterRight.SetLocation(ptStar + TiPoint(15, 0));
        mEmitterRight.SetVelocity(TiFPoint(mfWidthVelocity, 0.f));
        ptStar.y -= mSprRightNinjaStar.GetSize().cy / 2;
        //mSprRightNinjaStar.Draw(ptStar);
      }
      TiMenuItem& item = menuItems[i];
      if(item.mType == TiMenuItem::kCheck)
      {
        TiPoint ptCheck = rect.TopLeft();
        //ptCheck.x += 25;
        ptCheck.y += (rect.Height() - mSprUnchecked.GetSize().cy) / 2;
        mSprUnchecked.Draw(ptCheck);
        if(item.GetCheck())
        {
          ptCheck += TiPoint(2, -4);
          mSprChecked.Draw(ptCheck);
        }
        rect.left += mSprUnchecked.GetSize().cx;
      }
      font.Draw(item.mText, rect, crText, uTextFlags);

      rect += TiPoint(0, c_sizeSelection.cy);
      rect.left = ptFirstItem.x;
    }


    // prompt
    TiRect rectPrompt(575, 600, 975, 630);
    char szBuffer[100];
    if( mpCurrentMenu->mpParent == NULL )
    {
      mpTitanUI->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), SyHashResourceID("UiSelectPrompt"));
    }
    else
    {
      mpTitanUI->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), SyHashResourceID("UiSelectBackPrompt"));
    }
    font.Draw(szBuffer, rectPrompt, c_crWhite, TI_TF_HALIGN_CENTER|TI_TF_SHADOW);


    mEmitterLeft.Update(TiUI::Instance()->GetFrameElapse());
    //mEmitterLeft.Draw();
    mEmitterRight.Update(TiUI::Instance()->GetFrameElapse());
    //mEmitterRight.Draw();

    mpRaster->End2D();
  }


  // process timer
  SyTime time = TiUI::Instance()->GetTime();
  if(mTime == 0)
  {
    mTime = TiUI::Instance()->GetTime();
  }
  if( time - mTime > SYTIME_SECOND * 7 )
  {
    NextMode();
  }
}

//***********************************************************************
// ProcessButtonDown
//***********************************************************************

bool TiStartWindow::ProcessButtonDown( int32 DeviceId, int32 ButtonId )
{
  if(mMode == kBSDownloads)
  {
    mMode = kStartScreen;
    return true;
  }

  switch( TiWindow::GetButton( ButtonId ) )
  {
  case TiButtonMap::kUp:
    Prev();
    return( true );

  case TiButtonMap::kDown:
    Next();
    return( true );

  case TiButtonMap::kCircle:
    Back();
    return( true );

  case TiButtonMap::kX:
    Select();
    return( true );

  default:
    return( TiWindow::ProcessButtonDown( DeviceId, ButtonId ) );
  }
}

//***********************************************************************
// ProcessKeyDown
//***********************************************************************

bool TiStartWindow::ProcessKeyDown( int32 KeyCode )
{
  switch( KeyCode )
  {
  case SyInputEvent::KEYCODE_G:
    mpTitanUI->GetScene()->TogglePerformanceHUDGrid();
  	return( true );
  default:
    return( true );
  }
}

//***********************************************************************
// ProcessMessage
//***********************************************************************

void TiStartWindow::ProcessMessage( TiWindowMessage* pMessage )
{
  switch( pMessage->GetMessageId() )
  {
  case TIWINDOW_SETFOCUS:
    {
      TiFocusMessage* pMsg = static_cast<TiFocusMessage*>(pMessage);
      const TiWindow* pOld = pMsg->GetOldWindow();
      if( pOld != mpClassWindow && pOld != mpNameWindow && pOld != mpLoadGameWindow )
      {
        // gained first focus (not from child)
        mpCurrentMenu = &mMenuMain;
        mTime = TiUI::Instance()->GetTime();

        mNumPlayers = 1;
        mCampaign = true;
        mInternet = false;
        mClass = 0;
      }
    }
    return;

  case TIWINDOW_COMMAND:
    {
      TiCommandMessage* pMsg = static_cast<TiCommandMessage*>(pMessage);

      switch( pMsg->GetCommandId() )
      {
      case Id_Easy:
        mDifficulty = 0;
        BeginCharacterCreation();
        return;
      case Id_Medium:
        mDifficulty = 1;
        BeginCharacterCreation();
        return;
      case Id_Hard:
        mDifficulty = 2;
        BeginCharacterCreation();
        return;
      case Id_Expert:
        mDifficulty = 3;
        BeginCharacterCreation();
        return;
      case Id_Go:
        {
          TiStartMessage msg( this, TI_START_GO );
          mpTitanUI->ProcessCommandMessage( &msg );
        }
        return;
	  case Id_LoadGame:
		  {
			  mpLoadGameWindow->Init( true );
			  mpLoadGameWindow->Show( true );
			  mpLoadGameWindow->SetFocus();
			  mMode = kLoadGame;
		  }
		  return;
      case Id_Exit_Titan:
        {
          TiCommandMessage msg( this, TIWINDOW_COMMAND, TI_CMD_EXIT_TITAN, false );
          mpTitanUI->ProcessCommandMessage( &msg );
        }
        return;

      case Id_Language_English:
      case Id_Language_French:
      case Id_Language_German:
        OnSwitchLanguage(pMsg->GetCommandId());
        return;

      case Id_Downloads:
        mMode = kBSDownloads;
        return;

      default:
        TiWindow::ProcessMessage( pMessage );
        return;
      }
    }
    return;

  case TIWINDOW_CLOSE:
    {
      TiWindow* pSender = pMessage->GetSender();
      if( pSender == mpClassWindow )
      {
        // back to start menu
        mMode = kStartScreen;
        mpClassWindow->Show( false );
        this->SetFocus();
      }
      else if( pSender == mpNameWindow )
      {
        // back to class selection
        mpNameWindow->Show( false );
        mpClassWindow->Show( true );
        mpClassWindow->SetFocus();
      }
	  else if( pSender == mpLoadGameWindow )
	  {
		  // back to start menu
		  mMode = kStartScreen;
		  mpLoadGameWindow->Show( false );
		  this->SetFocus();
	  }
    }
    return;

  case TIWINDOW_CHARACTER_CREATE:
    ProcessCreateMessage( pMessage );
    return;

  default:
    TiWindow::ProcessMessage( pMessage );
    return;
  }
}
  
void TiStartWindow::OnSwitchLanguage(int32 iMenuCommand)
{
  int32 iLanguage = T4::EN_US;

  mMenuOptions.SetCheck(Id_Language_English, false);
  mMenuOptions.SetCheck(Id_Language_French, false);
  mMenuOptions.SetCheck(Id_Language_German, false);
  switch(iMenuCommand)
  {
  case Id_Language_English:
    iLanguage = T4::EN_US;
    break;
  case Id_Language_French:
    iLanguage = T4::FR_FR;
    break;
  case Id_Language_German:
    iLanguage = T4::DE_DE;
    break;
  }
  mMenuOptions.SetCheck(iMenuCommand, true);


  mpTitanUI->GetTitanI()->InitT4WithLanguage(iLanguage);
  LoadMenuText();
}

//***********************************************************************
// StartScreen
//  Go straight to menu
//***********************************************************************

void TiStartWindow::StartScreen()
{
  mMode = kStartScreen;
}

//***********************************************************************
// UpdateMenuWidth
//***********************************************************************
static float32 c_fInitialWidthVelocity = 200.f;
static float32 c_fAccel = 1200.f;
static float32 c_fFriction = 3.f;
static float32 c_fWidthEpsilon = 3.5f;

void TiStartWindow::UpdateMenuSelectionWidth()
{
  float32 fElapsed = SY_CLAMP(TiUI::Instance()->GetFrameElapse(), 0.001f, 0.02f);
  if(mfCurrentWidth < mfTargetWidth)
  {
    if(mfWidthVelocity < 0.f)
    {
      mfWidthVelocity += (c_fAccel * fElapsed);
      if(mfWidthVelocity > 0.f && mfCurrentWidth > mfTargetWidth - c_fWidthEpsilon)
      {
        mfWidthVelocity = 0.f;
        mfCurrentWidth = mfTargetWidth;
        mEmitterLeft.StopEmitting();
        mEmitterRight.StopEmitting();
      }
    }
    else
    {
      mfWidthVelocity += (c_fAccel * fElapsed);
    }
  }
  else if(mfCurrentWidth > mfTargetWidth)
  {
    if(mfWidthVelocity > 0.f)
    {
      mfWidthVelocity -= (c_fAccel * fElapsed);
      if(mfWidthVelocity < 0.f && mfCurrentWidth < mfTargetWidth + c_fWidthEpsilon)
      {
        mfWidthVelocity = 0.f;
        mfCurrentWidth = mfTargetWidth;
        mEmitterLeft.StopEmitting();
        mEmitterRight.StopEmitting();
      }
    }
    else
    {
      mfWidthVelocity -= (c_fAccel * fElapsed);
    }
  }
  else if(mfWidthVelocity == 0.f)
  {
    mEmitterLeft.StopEmitting();
    mEmitterRight.StopEmitting();
  }

  mfCurrentWidth += mfWidthVelocity * fElapsed;

  float32 fFrictionLoss = 1.f - c_fFriction * fElapsed;
  mfWidthVelocity *= fFrictionLoss;
}

//***********************************************************************
// OnNewMenuItem
//***********************************************************************
void TiStartWindow::OnNewMenuItem()
{
  TiFont font(TiUI::Instance()->GetFontHandle("Default"));
  std::vector< TiMenuItem >& menuItems = mpCurrentMenu->mMenuItems;
  TiMenuItem& item = menuItems[mpCurrentMenu->mSelection];
  mfTargetWidth = (float32)font.GetTextWidth(item.mText) + 40.f;

  if(mfTargetWidth < mfCurrentWidth)
  {
    mfWidthVelocity = -c_fInitialWidthVelocity;
  }
  else if(mfTargetWidth > mfCurrentWidth)
  {
    mfWidthVelocity = c_fInitialWidthVelocity;
  }
  mEmitterLeft.StartEmitting();
  mEmitterRight.StartEmitting();
}

//***********************************************************************
// Next
//***********************************************************************

void TiStartWindow::Next()
{
  if( mMode == kStartScreen && mpCurrentMenu )
  {
    mpCurrentMenu->Next();
    TiUI::Instance()->PlaySound( TiUI::kSoundTick );
    OnNewMenuItem();
  }
}

//***********************************************************************
// Prev
//***********************************************************************

void TiStartWindow::Prev()
{
  if( mMode == kStartScreen && mpCurrentMenu )
  {
    mpCurrentMenu->Prev();
    TiUI::Instance()->PlaySound( TiUI::kSoundTick );
    OnNewMenuItem();
  }
}

//***********************************************************************
// Back
//***********************************************************************

void TiStartWindow::Back()
{
  if( mMode == kStartScreen && mpCurrentMenu && mpCurrentMenu->mpParent != NULL )
  {
    // go up one menu
    mpCurrentMenu = mpCurrentMenu->mpParent;
    TiUI::Instance()->PlaySound( TiUI::kSoundBack );
  }
}

//***********************************************************************
// Select
//***********************************************************************

void TiStartWindow::Select()
{
  if( mMode == kEsrbLogo || mMode == kSoeLogo )
  {
    NextMode();
  }
  else if( mMode == kStartScreen )
  {
    if( mpCurrentMenu )
    {
      TiMenuItem& item = mpCurrentMenu->mMenuItems[ mpCurrentMenu->mSelection ];
      if( item.mpSubmenu != NULL )
      {
        // check ID
        switch( item.mId )
        {
        case Id_1Player:
          mNumPlayers = 1;
          break;
        case Id_2Players:
          mNumPlayers = 2;
        	break;
        case Id_3Players:
          mNumPlayers = 3;
          break;
        case Id_4Players:
          mNumPlayers = 4;
          break;
        case Id_Campaign:
          mCampaign = true;
          break;
        case Id_Arena:
          mCampaign = false;
          break;
        case Id_Normal:
          mInternet = false;
          break;
        case Id_Internet:
          mInternet = true;
          break;
        default:
          break;
        }

        // open sub menu
        mpCurrentMenu = item.mpSubmenu;
        TiUI::Instance()->PlaySound( TiUI::kSoundName );
      }
      else
      {
        TiUI::Instance()->PlaySound( TiUI::kSoundSelect );

        // send message to execute command
        TiCommandMessage msg( this, TIWINDOW_COMMAND, item.mId, !item.GetCheck() );
        ProcessMessage( &msg );
      }
    }
  }
}

//***********************************************************************
// NextMode
//***********************************************************************

void TiStartWindow::NextMode()
{
  mTime = TiUI::Instance()->GetTime();
  switch( mMode )
  {
  case kEsrbLogo:
    mMode = kSoeLogo;
    break;

  case kSoeLogo:
    {
      mMode = kStartScreen;
    }
    break;

  case kStartScreen:
  case kCharacter:
  default:
    break;
  }
}

//***********************************************************************
// ProcessCreateMessage
//***********************************************************************

void TiStartWindow::ProcessCreateMessage( TiWindowMessage* pMessage )
{
  TiCharacterCreateMessage* pMsg = static_cast<TiCharacterCreateMessage*>(pMessage);

  switch( pMsg->GetCharacterMessageId() )
  {
  case TI_CREATE_CLASS:
    // class has been selected, move to next screen
    mClass = pMsg->GetClass();
    mpClassWindow->Show( false );
    mpNameWindow->Show( true );
    mpNameWindow->SetFocus();
    return;

  case TI_CREATE_NAME:
    {
      // name has been selected, create new character & start game
      mpNameWindow->Show( false );

      TiStartMessage msg( this, TI_START_NEW_GAME );
      msg.SetCampaign( mCampaign );
      msg.SetClass( mClass );
      msg.SetDifficulty( mDifficulty );
      msg.SetInternet( mInternet );
      msg.SetName( pMsg->GetName() );
      msg.SetPlayers( mNumPlayers );
      mpTitanUI->ProcessCommandMessage( &msg );
    }
    return;
  }
}

//***********************************************************************
// BeginCharacterCreation
//***********************************************************************

void TiStartWindow::BeginCharacterCreation()
{
  // hide menus, splash screens, go to character creation
  mMode = kCharacter;
  mpClassWindow->Show( true );
  mpClassWindow->SetFocus();
}

void TiStartWindow::LoadMenuText()
{
  int32 iMenuIds[] = {
    Id_Go, Id_LoadGame, Id_Options, Id_Downloads, Id_Credits, Id_Exit_Titan,
    Id_Language_English, Id_Language_French, Id_Language_German,
    Id_Normal, Id_Internet,
    Id_1Player, Id_2Players, Id_3Players, Id_4Players,
    Id_Campaign, Id_Arena,
    Id_Easy, Id_Medium, Id_Hard, Id_Expert,
  };
  const char8 *pszStringIds[] = {
    "UiMainMenuStartGame", "UiMainMenuLoadGame", "UiMainMenuOptions", "UiMainMenuDownloads", "UiMainMenuCredits", "UiMainMenuExitTitan",
    "UiMainMenuEnglish", "UiMainMenuFrench", "UiMainMenuGerman",
    "UiMainMenuNormalGame", "UiMainMenuInternetGame",
    "UiMainMenu1Player", "UiMainMenu2Players", "UiMainMenu3Players", "UiMainMenu4Players",
    "UiMainMenuCampaign", "UiMainMenuArena",
    "UiMainMenuAdventurous", "UiMainMenuCourageous", "UiMainMenuChampion", "UiMainMenuLegendary",
  };
  TiMenu *pMenus[] = {
    &mMenuMain, &mMenuMain, &mMenuMain, &mMenuMain, &mMenuMain, &mMenuMain, 
    &mMenuOptions, &mMenuOptions, &mMenuOptions, 
    &mMenuNewGame, &mMenuNewGame, 
    &mMenuPlayers, &mMenuPlayers, &mMenuPlayers, &mMenuPlayers, 
    &mMenuGameType, &mMenuGameType, 
    &mMenuDifficulty, &mMenuDifficulty, &mMenuDifficulty, &mMenuDifficulty, 
  };

  for(int i = 0; i < (int)(sizeof(iMenuIds) / sizeof(int32)); i++)
  {
    SyResourceID id = SyHashResourceID(pszStringIds[i]);
    char szBuffer[100] = "";
    mpTitanUI->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), id);
    pMenus[i]->SetItemText(iMenuIds[i], szBuffer);
  }
}
