/********************************************************************

  Module: TiUIControl

  Author: Carlo Ordonez

  Description:

    Widget Drawing services

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#ifndef _TiUIControl_h_
#define _TiUIControl_h_

#include "SyTypes.h"
#include "SyVect2I.h"
#include "SyColor.h"
#include "SyString.h"
#include "SyTime.h"

// flags for text methods
#define TI_TF_SHADOW                (1<<0)    // draw text shadow
#define TI_TF_ELLIPSIS              (1<<1)    // print ellipses if text doesn't fit within box
#define TI_TF_HALIGN_CENTER         (1<<2)    // center horiz. within bounds
#define TI_TF_HALIGN_RIGHT          (1<<3)    // right justify within bounds
#define TI_TF_HARD_BREAK            (1<<4)    // break line at last possible character, not at last 'space' character; probably only useful for single-line text
#define TI_TF_FRAME                 (1<<5)    // draw frame around a text box
#define TI_TF_LINE_OFFSET(x)        ((x)<<24) // for multi-line textboxes, scroll down x lines

// text drawing escape sequences (see DrawTextBox())
//
//      |t - insert triangle button
//      |c - insert circle button
//      |x - insert 'x' button
//      |s - insert square button
//
//      |a - insert START button
//      |A - insert SELECT button
//
//      |l - insert L1 button
//      |L - insert L2 button
//      |r - insert R1 button
//      |R - insert R2 button
//
//      |z - insert up facing icon
//      |Z - insert down facing icon
//      |v - insert left facing icon
//      |V - insert right facing icon
//
//      || - insert pipe character

class TiUI;
class TiUILayout;

class TiUIControl
{
public:
  TiUIControl();
  ~TiUIControl();

  enum FrameStyle
  {
    kNone,
    kDefault, // thick metal border, gradient background
    kBlack,   // black background, semi-opaque
    kChild,   // flat, gray border, gradient background
  };
  enum TextColor
  {
    kNormal,
    kSelected,
    kDisabled,
    kTextColorMax
  };

  int32           Init( int32 Width, int32 NumRows, FrameStyle Style = kDefault );
  int32           Init( const SyVect2I& Center, int32 Width, int32 NumRows, FrameStyle Style = kDefault );
  int32           Init( const SyVect2I& Center, const SyVect2I& ClientSize, FrameStyle Style = kDefault );   

  void            Draw();
  void            UpdateFade();

  const SyVect2I  GetClientUL() const;
  const SyVect2I  GetClientLR() const;
  const SyVect2I  GetClientSize() const { return( mClientSize ); }
  const SyVect2I  GetRowCenter( int32 Row );

  bool            IsOpening() const;
  bool            IsOpen() const;
  void            Open( bool Instant = false );

  bool            IsClosing() const;
  bool            IsClosed() const;
  void            Close( bool Instant = false );

  float           GetOpacity() const { return( mFade ); }
  void            SetOpacity( float Opacity );

  void            SetFrameStyle( FrameStyle Style ) { mStyle = Style; }

  int32           DrawTextBox( const SyString& Text, int32 X, int32 FirstRow, int32 Width, int32 RowCount, const SyColor32F& Color, uint32 Flags );
  int32           DrawTextBox( const SyString& Text, int32 X, int32 FirstRow, int32 Width, int32 RowCount, TextColor Color, uint32 Flags );
  int32           GetTextWidth( const SyString& Text );

  int32           DrawTextBox( const char8* pText, int32 X, int32 FirstRow, int32 Width, int32 RowCount, const SyColor32F& Color, uint32 Flags );
  int32           DrawTextBox( const char8* pText, int32 X, int32 FirstRow, int32 Width, int32 RowCount, TextColor Color, uint32 Flags );
  int32           GetTextWidth( const char8* pText );

  void            DrawProgressBar( int32 X, int32 Row, int32 Width, float32 Progress );
  void            DrawVScroll( int32 X, int32 Row, int32 Width, int32 RowCount, float32 ScrollPos );

  float32         GetTickPulse() const;
  void            ResetTickPulse();

  // # of lines resulting from line wrapping within the given width
  int32           CountLines( const char8* pText, int32 Width );

  void            ReloadLayout();

private:

  void            AppendEllipsis( char8* pText, int32 MaxWidth );
  void            DrawTextInternal( const char8* pText, int32 X, int32 Y, const SyColor32F& Color, uint32 Flags );

  void            DrawFrame( int32 X, int32 FirstRow, int32 Width, int32 RowCount, uint32 Flags = 0 );
  void            InitColors();

private:

  TiUI*           mpUI;
  float32         mFade;
  int32           mFadeDir;
  SyTime          mTime;
  FrameStyle      mStyle;

  int32           mRows;
  SyVect2I        mCenter;
  SyVect2I        mClientSize;

  SyColor32F      mTextColors[ kTextColorMax ];

  TiUILayout*     mpDefaultLayout;
  TiUILayout*     mpChildLayout;
};

#endif

