/********************************************************************

  Module: TiUILayout

  Author: Carlo Ordonez

  Description:

    Layout information for UI windows

  Copyright 2006 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiUILayout.h"
#include "tinyxml/tinyxml.h"
#include "SyPathname.h"

//***********************************************************************
// Constructors/Destructors
//***********************************************************************

TiUILayoutItem::TiUILayoutItem()
{
  mName.Init( "" );
  mFilename.Init( "" );
}

TiUILayout::TiUILayout()
{
}

TiUILayout::~TiUILayout()
{
}

//***********************************************************************
// Init
//***********************************************************************

int32 TiUILayout::Init( const char8* pFilename )
{
  // clear previous contents
  mName.Init( "" );
  mItems.Clear();

  mFilename = pFilename;

  // open file
  SyString path = "programming\\ui\\";
  path += mFilename;
  SyPathname pathname( path.AsChar() );

  TiXmlDocument doc( pathname.Full().AsChar() );
  if( !doc.LoadFile() )
  {
    return( -1 );
  }

  TiXmlHandle root( &doc );

  const char8* pDocName = pathname.Base().AsChar();
  mName = pDocName;

  // read items
  for( TiXmlElement* pElement = root.FirstChildElement(pDocName).FirstChildElement("image").Element();
       pElement;
       pElement = pElement->NextSiblingElement( "image" ) )
  {
    TiUILayoutItem newItem;
    newItem.mName = pElement->Attribute( "name" );
    newItem.mFilename = pElement->Attribute( "file" );
    pElement->QueryIntAttribute( "x", &newItem.mOffset.X );
    pElement->QueryIntAttribute( "y", &newItem.mOffset.Y );
    pElement->QueryIntAttribute( "width", &newItem.mSize.X );
    pElement->QueryIntAttribute( "height", &newItem.mSize.Y );

    mItems.Insert( newItem.mName, newItem );
  }
  return( 0 );
}

//***********************************************************************
// Reload
//***********************************************************************

int32 TiUILayout::Reload()
{
  return( Init( mFilename.AsChar() ) );
}

//***********************************************************************
// GetOffset
//***********************************************************************

const SyVect2I TiUILayout::GetOffset( const char8* pImageName )
{
  int32 find = mItems.Find( SyString(pImageName) );
  SyAssertf( find != mItems.End(), "Image not found" );
  if( find != mItems.End() )
  {
    TiUILayoutItem& item = mItems(find);
    return( item.mOffset );
  }
  else
  {
    // not found
    return( SyVect2I(0,0) );
  }
}

//***********************************************************************
// GetLayoutInfo
//***********************************************************************

int32 TiUILayout::GetLayoutInfo( const char8* pImageName, TiUILayoutItem* pInfo )
{
  int32 find = mItems.Find( SyString(pImageName) );
  SyAssertf( find != mItems.End(), "Image not found" );
  if( find != mItems.End() )
  {
    if( pInfo )
    {
      *pInfo = mItems(find);
    }
    return( 0 );
  }
  else
  {
    // not found
    return( -1 );
  }
}
