/********************************************************************

  Module: TiWindow

  Author: Carlo Ordonez

  Description:

    Window base class

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiWindow.h"
#include "TiUI.h"
#include "SyRaster.h"
#include "TiFocusMessage.h"
#include "TiApp.h"

TiButtonMap TiWindow::mButtonMap;

//***********************************************************************
// Constructor
//***********************************************************************
TiWindow::TiWindow( TiApp* pApp )
{
  mpApp = pApp;
  mpParent = 0;
  mVisible = false;
  m_bRelativeToParent = true;
  m_rect.SetRect(0, 0, 0, 0);
}

//***********************************************************************
// Constructor
//***********************************************************************

TiWindow::TiWindow( TiWindow* pParent )
{
  mpApp = pParent->mpApp;
  mVisible = false;
  pParent->AddChild( this );
}

//***********************************************************************
// Destructor
//***********************************************************************

TiWindow::~TiWindow()
{
  if (mpApp->GetFocus() == this)
  {
      // if we are the current focus window, then clear the focus, because we are getting destroyed
    mpApp->SetFocus(NULL);
  }

  if ( mpParent )
  {
    // let the parent know we're going
    mpParent->RemoveChild( this );
  }

  std::set< TiWindow* >::iterator it;
  for( it = mChildren.begin(); it != mChildren.end(); it = mChildren.begin() )
  {
    TiWindow* pChild = *it;
    delete pChild;
  }
}

//***********************************************************************
// AddChild
//***********************************************************************

void TiWindow::AddChild( TiWindow* pChild )
{
  pChild->mpParent = this;
  SyAssert( mChildren.find( pChild ) == mChildren.end() );

  mChildren.insert( pChild );
}

//***********************************************************************
// RemoveChild
//***********************************************************************

void TiWindow::RemoveChild( TiWindow* pChild )
{
  SyAssert( pChild->mpParent == this );

  std::set< TiWindow* >::iterator it = mChildren.find( pChild );
  if( it != mChildren.end() )
  {
    mChildren.erase( it );
    pChild->mpParent = NULL;
  }
}

//***********************************************************************
// ProcessFrame
//***********************************************************************

void TiWindow::ProcessFrame()
{
  OnProcessFrame();
  // process children
  std::set< TiWindow* >::const_iterator it;
  for( it = mChildren.begin(); it != mChildren.end(); it++ )
  {
    TiWindow* pChild = *it;
    pChild->ProcessFrame();
  }
}


//***********************************************************************
// Draw
//***********************************************************************

void TiWindow::Draw()
{
  if( this->IsVisible() )
  {
    DrawSelf();

    // draw children
    std::set< TiWindow* >::const_iterator it;
    for( it = mChildren.begin(); it != mChildren.end(); it++ )
    {
      TiWindow* pChild = *it;
      if( pChild->IsVisible() )
      {
        pChild->Draw();
      }
    }
  }
}

//***********************************************************************
// ProcessKeyDown
//***********************************************************************

bool TiWindow::ProcessAxisMove( int32 iDevice, int32 x, int32 y, int32 nButtonStates )
{
  return false;
}

//***********************************************************************
// ProcessKeyDown
//***********************************************************************

bool TiWindow::ProcessKeyDown( int32 KeyCode )
{
  return false;
}

//***********************************************************************
// ProcessKeyUp
//***********************************************************************

bool TiWindow::ProcessKeyUp( int32 KeyCode )
{
  return false;
}

//***********************************************************************
// ProcessButtonDown
//***********************************************************************

bool TiWindow::ProcessButtonDown( int32 DeviceId, int32 ButtonId )
{
  return false;
}

//***********************************************************************
// ProcessButtonUp
//***********************************************************************

bool TiWindow::ProcessButtonUp( int32 DeviceId, int32 ButtonId )
{
  return false;
}

//***********************************************************************
// ProcessMessage
//***********************************************************************

void TiWindow::ProcessMessage( TiWindowMessage* pMessage )
{
  if( mpParent != NULL )
  {
    mpParent->ProcessMessage( pMessage );
  }
}

//***********************************************************************
// FindFocus
//***********************************************************************

TiWindow* TiWindow::FindFocus()
{
  return mpApp->GetFocus();
}

//***********************************************************************
// SetFocus
//***********************************************************************

bool TiWindow::SetFocus()
{
  // do the change
  return mpApp->SetFocus( this );
}

//***********************************************************************
// GetParent
//***********************************************************************

TiWindow* TiWindow::GetParent() const
{
  return mpParent;
}

//***********************************************************************
// SetButtonMap
//***********************************************************************

void TiWindow::SetButtonMap( TiButtonMap::Mapping Mapping )
{
  mButtonMap.SetMapping( Mapping );
}

//***********************************************************************
// GetButton
//***********************************************************************

TiButtonMap::Button TiWindow::GetButton( int32 ButtonId )
{
  return( mButtonMap.GetButton( ButtonId ) );
}

//***********************************************************************
// CloseWindow
//***********************************************************************

void TiWindow::CloseWindow()
{
  // send request to parent
  TiWindowMessage msg( this, TIWINDOW_CLOSE );
  ProcessMessage( &msg );
}

//***********************************************************************
// ReloadLayout
//***********************************************************************

void TiWindow::ReloadLayout()
{
  // self
  ProcessReloadLayout();

  // children
  std::set< TiWindow* >::const_iterator it;
  for( it = mChildren.begin(); it != mChildren.end(); it++ )
  {
    TiWindow* pChild = *it;
    pChild->ProcessReloadLayout();
  }
}

void TiWindow::Move(TiPoint pt)
{
  m_rect.MoveTo(pt);
}

void TiWindow::Move(TiRect rect)
{
  m_rect = rect;
}

void TiWindow::SetCenter(TiPoint pt)
{
  TiSize size = m_rect.GetSize();

  pt.x -= size.cx / 2;
  pt.y -= size.cy / 2;

  Move(pt);
}

TiRect TiWindow::GetScreenRect() const
{
  TiRect rect = m_rect;

  const TiWindow *pParent = GetParent();
  if(pParent != NULL)
  {
    rect.MoveBy(pParent->GetScreenRect().TopLeft());
  }
  return rect;
}

