/*   SCE CONFIDENTIAL                                       */
/*   PLAYSTATION(R)3 Programmer Tool Runtime Library 084.006 */
/*   Copyright (C) 2006 Sony Computer Entertainment Inc.    */
/*   All Rights Reserved.                                   */

/**
 * \defgroup all_syscall System Calls
 */

/**
 * \ingroup all_syscall
 * \defgroup proc Process Management
 * \brief system calls defined in sys/process.h
 */
/*@{*/

#ifndef __LV2_KERNEL_PROC_SYSCALL_PROCESS_H__
#define __LV2_KERNEL_PROC_SYSCALL_PROCESS_H__

#include <sys/cdefs.h>
#include <sys/syscall.h>
#include <sys/types.h>
#include <sys/integertypes.h>
#include <sys/return_code.h>

CDECL_BEGIN



#define SYS_PROCESS_PARAM_SECTION_NAME		".sys_proc_param"
#define SYS_PROCESS_PARAM_INVALID_PRIO		-32768
#define SYS_PROCESS_PARAM_INVALID_STACK_SIZE	0xffffffffUL

#define SYS_PROCESS_PARAM_STACK_SIZE_MIN	0x1000	// 4KB
#define SYS_PROCESS_PARAM_STACK_SIZE_MAX	0x100000	// 1MB

#define SYS_PROCESS_PARAM_VERSION_INVALID	0xffffffff
#define SYS_PROCESS_PARAM_VERSION_1		0x00000001

/* \brief process parameters */
typedef struct {
	uint32_t version;
	int32_t primary_prio;
	uint32_t primary_stacksize;
	uint32_t reserved;
} sys_process_param_t;

#define SYS_PROCESS_PARAM_SECTION \
	__attribute__((aligned(8), section(SYS_PROCESS_PARAM_SECTION_NAME), unused))

#define SYS_PROCESS_PARAM( primary_prio, primary_stacksize ) \
	sys_process_param_t __sys_process_param SYS_PROCESS_PARAM_SECTION = { \
		SYS_PROCESS_PARAM_VERSION_1, primary_prio, primary_stacksize, 0 };

/** \brief memory mapping in the process address space (32-bit mode)
 * <PRE>
 *
 *   0xFFFF_FFFF           +-------------------------------+
 *                         | SPU Thread Mapping Area       |
 *   SPU_THREAD_BASE_ADDR  +-------------------------------+
 *                         | Raw SPU Mapping Area          |
 *   RAW_SPU_BASE_ADDR     +-------------------------------+
 *                         | User Area                     |
 *   0xD000_0000           +-------------------------------+
 *                         | RSX Frame Buffer Mapping Area |
 *   RSX_FB_BASE_ADDR      +-------------------------------+
 *                         |                               |
 *                         | User Area (heap, ...)         |
 *                         |                               |
 *                         |                               |
 *                         |                               |
 *                         +-------------------------------+
 *                         | PPU/SPU/RSX Shared Segment    |
 *                         | (.rsx_image)                  |
 *                         +-------------------------------+
 *                         | PPU/SPU Shared Segment        |
 *                         | (.rodata, .data, .bss, ...)   |
 *                         +-------------------------------+
 *                         | PPU Local Segment             |
 *                         | (.text, .ppu_rodata, ...)     |
 *  TEXT_SEGMENT_BASE_ADDR +-------------------------------+
 *                         | Unmapped Area                 |
 *  0x0000_0000            +-------------------------------+
 *
 * </PRE>
 */

/** \brief  get the process id */
static inline sys_pid_t sys_process_getpid(void)
{
	system_call(0, SYS_PROCESS_GETPID);

#ifdef REG_PASS_SYS_PROCESS_GETPID
	REG_PASS_SYS_PROCESS_GETPID;
#endif
	return_to_user_prog(sys_pid_t);
}


/** \brief get the parent process id */
static inline sys_pid_t sys_process_getppid(void)
{
	system_call(0, SYS_PROCESS_GETPPID);

#ifdef REG_PASS_SYS_PROCESS_GETPPID
	REG_PASS_SYS_PROCESS_GETPPID;
#endif
	return_to_user_prog(sys_pid_t);
}



/** \brief terminate a process */
static inline void sys_process_exit(int status)
{
	system_call(1, SYS_PROCESS_EXIT, status);

#ifdef REG_PASS_SYS_PROCESS_EXIT
	REG_PASS_SYS_PROCESS_EXIT;
#endif
	return_to_user_prog(void);
}



CDECL_END

#endif /* __LV2_KERNEL_PROC_SYSCALL_PROCESS_H__ */
/*@}*/
