/******************************************************************
  
  Module:  rule.cpp
  
  Author: Sean Craig
                    
  Copyright 2005 Sony Online Entertainment.  All rights reserved.
  
*******************************************************************/

//-------------------------------------------------------- Includes
#include "rule.h"
#include "rule_condition.h"
#include "titan.h"
#include "registry.h"
//---------------------------------------------- Class Declarations

//--------------------------------------------------------- Globals
static bool l_logEnabled=false;
static int  l_logIndent = 0;
static int  l_logHiddenLevel = 0; // indent level of hidden log
//----------------------------------------- Functions Declarations
//------------------------------------ Member Functions Definitions

cRule::cRule() :
  mDisabled(false),
  mBlocked(false),
  mpTitan(NULL),
  mSpellID(ID_NONE),
  mSourceID(ID_NONE)
{
}

void 
cRule::ProcessPacket(cRulePacket *packet)
{
  SyAssertf(mpTitan != NULL, "Rule needs to have titan ptr set.");
  if (mDisabled)
  {
    return;
  }

  SyAssert(packet != NULL);
  SyAssertf(mpTitan != NULL,"Rules must be added to a rule container before being used");


  switch (packet->mPacketType)
  {
    case cRulePacket::CALC_HEALTH_MAX:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_HEALTH_MAX bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcHealthMax(calcPacket);
    }
    break;
    case cRulePacket::CALC_HEALTH_REGEN:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_HEALTH_REGEN bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcHealthRegen(calcPacket);
    }
    break;
    case cRulePacket::CALC_MANA_MAX:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_MANA_MAX bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcManaMax(calcPacket);
    }
    break;
    case cRulePacket::CALC_MANA_REGEN:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_MANA_REGEN bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcManaRegen(calcPacket);
    }
    break;
    case cRulePacket::CALC_BLOCK_MAX:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_BLOCK_MAX bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcBlockMax(calcPacket);
    }
    break;
    case cRulePacket::CALC_BLOCK_REGEN:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_BLOCK_REGEN bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcBlockRegen(calcPacket);
    }
    break;
    case cRulePacket::CALC_STRIKETHROUGH:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_DAMAGE, "CALC_STRIKETHROUGH bad packet type");
      cDamagePacket *damagePacket = (cDamagePacket*)packet;
      CalcStrikethrough(damagePacket);
    }
    break;
    case cRulePacket::CALC_DAMAGE_INITIAL:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_DAMAGE, "CALC_DAMAGE_INITIAL bad packet type");
      cDamagePacket *damagePacket = (cDamagePacket*)packet;
      CalcDamageInitial(damagePacket);
    }
    break;
    case cRulePacket::CALC_DAMAGE_RAW:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_DAMAGE, "CALC_DAMAGE_RAW bad packet type");
      cDamagePacket *damagePacket = (cDamagePacket*)packet;
      CalcDamageRaw(damagePacket);
    }
    break;
    case cRulePacket::CALC_DAMAGE_REDUCED:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_DAMAGE, "CALC_DAMAGE_REDUCED bad packet type");
      cDamagePacket *damagePacket = (cDamagePacket*)packet;
      CalcDamageReduced(damagePacket);
    }
    break;
    case cRulePacket::CALC_CRITICAL_CHANCE:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_DAMAGE, "CALC_CRITICAL_CHANCE bad packet type");
      cDamagePacket *damagePacket = (cDamagePacket*)packet;
      CalcCriticalChance(damagePacket);
    }
    break;
    case cRulePacket::CALC_DODGE_CHANCE:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_DAMAGE, "CALC_DODGE_CHANCE bad packet type");
      cDamagePacket *damagePacket = (cDamagePacket*)packet;
      CalcDodgeChance(damagePacket);
    }
    break;
    case cRulePacket::CALC_OFFENSE:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_DAMAGE, "CALC_OFFENSE bad packet type");
      cDamagePacket *damagePacket = (cDamagePacket*)packet;
      CalcOffense(damagePacket);
    }
    break;
    case cRulePacket::CALC_DEFENSE:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_DAMAGE, "CALC_DEFENSE bad packet type");
      cDamagePacket *damagePacket = (cDamagePacket*)packet;
      CalcDefense(damagePacket);
    }
    break;
    case cRulePacket::CALC_SPELL_POWER:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_SPELL_POWER bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcSpellPower(calcPacket);
    }
    break;
    case cRulePacket::CALC_EXPERIENCE_GAIN:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_EXPERIENCE_GAIN bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcExperienceGain(calcPacket);
    }
    break;
    case cRulePacket::CALC_ATTACK_SPEED:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_ATTACK_SPEED bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcAttackSpeed(calcPacket);
    }
    break;
    case cRulePacket::CALC_MOVEMENT_SPEED:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_MOVEMENT_SPEED bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcMovementSpeed(calcPacket);
    }
    break;
    case cRulePacket::CALC_HEALTH_ABSORB:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_HEALTH_ABSORB bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcHealthAbsorb(calcPacket);
    }
    break;
    case cRulePacket::CALC_MANA_ABSORB:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_MANA_ABSORB bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcManaAbsorb(calcPacket);
    }
    break;
    case cRulePacket::CALC_ESSENCE_ABSORB:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_ESSENCE_ABSORB bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcEssenceAbsorb(calcPacket);
    }
    break;
    case cRulePacket::CALC_HEALTH_DROP_CHANCE:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_HEALTH_DROP_CHANCE bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcHealthDropChance(calcPacket);
    }
    break;
    case cRulePacket::CALC_MANA_DROP_CHANCE:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_MANA_DROP_CHANCE bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcManaDropChance(calcPacket);
    }
    break;
    case cRulePacket::CALC_ESSENCE_DROP_CHANCE:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_ESSENCE_DROP_CHANCE bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcEssenceDropChance(calcPacket);
    }
    break;
    case cRulePacket::CALC_ITEM_DROP_CHANCE:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "CALC_ITEM_DROP_CHANCE bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket*)packet;
      CalcItemDropChance(calcPacket);
    }
    break;
    case cRulePacket::EVENT_HIT:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_HIT || packet->GetClass() == cRulePacket::CLASS_DAMAGE, "EVENT_HIT bad packet type");
      cHitPacket *hitPacket = (cHitPacket*)packet;
      OnHit(hitPacket);
    }
    break;
    case cRulePacket::EVENT_APPLY_DAMAGE:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_DAMAGE, "EVENT_APPLY_DAMAGE bad packet type");
      cDamagePacket *damagePacket = (cDamagePacket *)packet;
      OnApplyDamage(damagePacket);
    }
    break;
    case cRulePacket::EVENT_APPLY_HEALING:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "EVENT_APPLY_HEALING bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket *)packet;
      OnApplyHealing(calcPacket);
    }
    break;
    case cRulePacket::EVENT_APPLY_MANA_COST:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_CALC, "EVENT_APPLY_MANA_COST bad packet type");
      cCalcPacket *calcPacket = (cCalcPacket *)packet;
      OnApplyManaCost(calcPacket);
    }
    break;
    case cRulePacket::QUERY_FLAG:
    { 
      SyAssertf(packet->GetClass() == cRulePacket::CLASS_FLAG, "QUERY_FLAG bad packet type");
      cFlagPacket *flagPacket = (cFlagPacket *)packet;
      QueryFlag(flagPacket);
    }
    break;
    default:
      SyAssert(0); // unknown packet
    break;
  }
}

void 
cRule::SetDisabled()
{
  mDisabled = true;
}

void 
cRule::SetBlocked(bool bBlocked)
{
  mBlocked = bBlocked;
}
//------------------------------------ cRuleContainer

  
cRuleContainer::cRuleContainer() : cRule(),
  mOwnerID(ID_NONE)
{
};

cRuleContainer::~cRuleContainer()
{
  Clear();
}

void
cRuleContainer::SetOwnerID(tGameObjectID ownerID)
{
  mOwnerID = ownerID;
}

bool    
cRuleContainer::Add(cRule *rule)
{
  SyAssertf(mpTitan != NULL,"Must set pointer to titan before adding rules to container");

  // first, allow other rules to combine with the new rule (for example, same spell cast twice : some rules don't stack)
  for (int ii=mRules.Begin(); ii!=mRules.End(); ii=mRules.Next(ii))
  {
    if (mRules(ii)->Combine(rule))
    {
      delete rule; // must delete it now to not leak mem (container always assumes ownership 
      return false;
    }

    if (mRules(ii)->Block(rule))
    {
      rule->SetBlocked(true);
    }
  }
  rule->SetTitan(mpTitan);
  mRules.Add(rule);
  rule->OnAdded(mOwnerID);
  return true;
}

bool    
cRuleContainer::Remove(const char *name, tGameID spellID, tGameObjectID sourceID, tGameID itemMasterID)
{
  bool retval = false;
  for (int ii=mRules.Begin(); ii!=mRules.End(); ii=mRules.Next(ii))
  {
    if ((ID_NONE == spellID || mRules(ii)->GetSpellID() == spellID) &&
        (ID_NONE == sourceID || mRules(ii)->GetSourceID() == sourceID) &&
        (ID_NONE == itemMasterID || mRules(ii)->GetItemMasterID() == itemMasterID))
    {
      if (SyStr::Stricmp(mRules(ii)->GetName(),name)==0)
      {
        mRules(ii)->SetDisabled(); // will actually get removed next update
        retval = true;
      }
    }
  }
  return retval;
}

bool    
cRuleContainer::RemoveAll(tGameID spellID, tGameObjectID sourceID, tGameID itemMasterID)
{
  bool retval = false;
  for (int ii=mRules.Begin(); ii!=mRules.End(); ii=mRules.Next(ii))
  {
    if ((ID_NONE == spellID || mRules(ii)->GetSpellID() == spellID) &&
        (ID_NONE == sourceID || mRules(ii)->GetSourceID() == sourceID) &&
        (ID_NONE == itemMasterID || mRules(ii)->GetItemMasterID() == itemMasterID))
    {
      mRules(ii)->SetDisabled(); // will actually get removed next update
      retval = true;
    }
  }
  return retval;
}

void    
cRuleContainer::Clear()
{
  for (int ii=mRules.Begin(); ii!=mRules.End(); ii=mRules.Next(ii))
  {
    delete mRules(ii);
  }
  mRules.Clear();
}

void    
cRuleContainer::Update(float time)
{
  for (int ii=mRules.Begin(); ii!=mRules.End(); ii=mRules.Next(ii))
  {
    if (!mRules(ii)->IsBlocked())
    {
      mRules(ii)->Update(time);
    }
  }
  ClearDisabled();
}

void
cRuleContainer::ClearDisabled()
{
  bool bRuleRemoved = false;

  for (int ii=mRules.Begin(); ii!=mRules.End();)
  {
    if (mRules(ii)->IsDisabled())
    {
      bRuleRemoved = true;
      mRules(ii)->OnRemoved(mOwnerID);
      delete mRules(ii);

      // keep the rules in order of addition when deleted
      for (int jj=ii; jj!=mRules.End(); jj=mRules.Next(jj))
      {
        if (mRules.Next(jj) != mRules.End())
        {
          mRules(jj) = mRules(mRules.Next(jj));
        }
      }

      mRules.Erase();

      if (ii >= mRules.Size())
      {
        ii = mRules.End();
      }
    }
    else
    {
      ii=mRules.Next(ii);
    }
  }

  if (bRuleRemoved)
  {
    ResetBlocked();
  }
}

void
cRuleContainer::ResetBlocked()
{
  for (int ii=mRules.Begin(); ii!=mRules.End(); ii=mRules.Next(ii))
  {
    mRules(ii)->SetBlocked(false);
  }

  for (int ii=mRules.Begin(); ii!=mRules.End(); ii=mRules.Next(ii))
  {
    if (!mRules(ii)->IsBlocked())
    {
      for (int jj=mRules.Next(ii); jj!=mRules.End(); jj=mRules.Next(jj))
      {
        if (mRules(ii)->Block(mRules(jj)))
        {
          mRules(jj)->SetBlocked(true);
        }
      }
    }
  }
}

 
void  
cRuleContainer::ProcessPacket(cRulePacket *packet)
{
  for (int ii=mRules.Begin(); ii!=mRules.End(); ii=mRules.Next(ii))
  {
    if (!mRules(ii)->IsBlocked())
    {
      mRules(ii)->ProcessPacket(packet);
    }
  }
}

bool    
cRuleContainer::Contains(const char *name)
{
  for (int ii=mRules.Begin(); ii!=mRules.End(); ii=mRules.Next(ii))
  {
    if (!mRules(ii)->IsDisabled() && SyStr::Stricmp(mRules(ii)->GetName(),name)==0)
    {
      return true;
    }
  }
  return false;
}


//-------------------------------------------------------------------------------------------- Global functions


void 
LogEnable(bool log)
{
  if (log == true && l_logEnabled == false)
  {
    static const char *filename = "log.txt";
    // clear out log file...
    FILE *logfile = fopen(filename,"w");
    fclose(logfile);
  }
  l_logEnabled = log;
}

bool 
IsLogEnabled()
{
  return l_logEnabled;
}

void 
Log(const char *format,...)
{
  if (l_logEnabled && l_logIndent < l_logHiddenLevel)
  {
    static const char *filename = "log.txt";
    // 
    FILE *logfile = fopen(filename,"a+");
    if (logfile != NULL)
    {
      va_list ap;
      va_start (ap,format);

      for (int ii=0;ii<l_logIndent;++ii)
      {
        fprintf(logfile,"\t");
      }
      vfprintf(logfile,format,ap);
      va_end(ap);
      fclose(logfile);
    }
  }
}

const char *l_PacketNames[] = 
{
  "CALC_HEALTH_MAX",
  "CALC_HEALTH_REGEN",
  "CALC_MANA_MAX",
  "CALC_MANA_REGEN",
  "CALC_DAMAGE_INITIAL", // called on Attacker
  "CALC_DAMAGE_RAW",     // called on Attacker
  "CALC_DAMAGE_REDUCED", // called on Defender
  "CALC_BLOCK_MAX",      // called on Defender
  "CALC_BLOCK_REGEN",
  "CALC_STRIKETHROUGH",  // called on Attacker

  "CALC_CRITICAL_CHANCE",
  "CALC_DODGE_CHANCE",
  "CALC_OFFENSE",
  "CALC_DEFENSE",
  "CALC_SPELL_POWER",
  "CALC_CARRYING_CAPACITY",
  "CALC_EXPERIENCE_GAIN",
  "CALC_ATTACK_SPEED",
  "CALC_MOVEMENT_SPEED",
  "CALC_HEALTH_ABSORB",
  "CALC_MANA_ABSORB",
  "CALC_ESSENCE_ABSORB",
  "CALC_HEALTH_DROP_CHANCE",
  "CALC_MANA_DROP_CHANCE",
  "CALC_ESSENCE_DROP_CHANCE",
  "CALC_ITEM_DROP_CHANCE",

  "QUERY_FLAG",

  "EVENT_HIT",               // hit someone else (called on Attacker)
  "EVENT_APPLY_DAMAGE",      // took damage (called on Defender)
  "EVENT_APPLY_HEALING",      // took damage (called on Defender)
  "EVENT_APPLY_MANA_COST",   // cast spell
};

#define NUM_PACKET_NAMES  (sizeof(l_PacketNames)/sizeof(l_PacketNames[0]))

void 
LogPacketStart(cRulePacket *packet, Titan *titan)
{

  SyAssertf(NUM_PACKET_NAMES == cRulePacket::NUM_PACKET_TYPES, "Forgot to name a packet type?");


  switch(packet->mPacketType)
  {
    // these are the packets we want to log
    case cRulePacket::CALC_DAMAGE_INITIAL:
    case cRulePacket::CALC_DAMAGE_RAW:
    case cRulePacket::CALC_DAMAGE_REDUCED:
    case cRulePacket::CALC_STRIKETHROUGH:
    case cRulePacket::CALC_OFFENSE:
    case cRulePacket::CALC_DEFENSE:
      break;
    default:
    // don't log these packets
      l_logHiddenLevel = l_logIndent;
      ++l_logIndent;
      return;
  }
  
  if (l_logEnabled && l_logIndent < l_logHiddenLevel)
  {

    cGameObject *obj = titan->GetRegistry()->Fetch(packet->mObjectID);
    const char *name = "Unknown";
    if (obj != NULL && obj->GetName() != NULL && obj->GetName()[0] != '\0')
    {
      name = obj->GetName();
    }
    switch(packet->GetClass())
    {
    case cRulePacket::CLASS_BASE:
      Log("Enter %s: %s\n",l_PacketNames[packet->mPacketType],name);
      break;
    case cRulePacket::CLASS_CALC:
      Log("Enter %s: %s Initial Value: %d\n",l_PacketNames[packet->mPacketType],name,((cCalcPacket*)packet)->GetTotal());
      break;
    case cRulePacket::CLASS_HIT:
      {
        cGameObject *attacker = titan->GetRegistry()->Fetch(((cHitPacket*)packet)->mAttackerID);
        const char *attacker_name = "Unknown";
        if (attacker != NULL && attacker->GetName() != NULL && attacker->GetName()[0] != '\0')
        {
          attacker_name = attacker->GetName();
        }
        cGameObject *defender = titan->GetRegistry()->Fetch(((cHitPacket*)packet)->mDefenderID);
        const char *defender_name = "Unknown";
        if (defender != NULL && defender->GetName() != NULL && defender->GetName()[0] != '\0')
        {
          defender_name = defender->GetName();
        }
        Log("Enter %s: %s; %s attacks %s; Initial Value: %d\n",
            l_PacketNames[packet->mPacketType],name,attacker_name,defender_name,((cCalcPacket*)packet)->GetTotal());
      }

      break;
    case cRulePacket::CLASS_DAMAGE:
      {
        cGameObject *attacker = titan->GetRegistry()->Fetch(((cHitPacket*)packet)->mAttackerID);
        const char *attacker_name = "Unknown";
        if (attacker != NULL && attacker->GetName() != NULL && attacker->GetName()[0] != '\0')
        {
          attacker_name = attacker->GetName();
        }
        cGameObject *defender = titan->GetRegistry()->Fetch(((cHitPacket*)packet)->mDefenderID);
        const char *defender_name = "Unknown";
        if (defender != NULL && defender->GetName() != NULL && defender->GetName()[0] != '\0')
        {
          defender_name = defender->GetName();
        }
        Log("Enter %s: %s; %s attacks %s; Initial Value: %d\n",
            l_PacketNames[packet->mPacketType],name,attacker_name,defender_name,((cCalcPacket*)packet)->GetTotal());
      }
      break;
    case cRulePacket::CLASS_FLAG:
      {
      }
      break;
    default:
      SyAssertf(0,"Unknown packet type");
    }

  }
  ++l_logIndent;
}

void 
LogPacketEnd(cRulePacket *packet,Titan *titan)
{
  if (l_logEnabled)
  {
    --l_logIndent;
      
    if (l_logIndent < l_logHiddenLevel)
    {
      cGameObject *obj = titan->GetRegistry()->Fetch(packet->mObjectID);
      const char *name = "Unknown";
      if (obj != NULL && obj->GetName() != NULL && obj->GetName()[0] != '\0')
      {
        name = obj->GetName();
      }
      switch(packet->GetClass())
      {
      case cRulePacket::CLASS_BASE:
        Log("Exit %s: %s\n",l_PacketNames[packet->mPacketType],name);
        break;
      case cRulePacket::CLASS_CALC:
        Log("Exit %s: %s Total: %d\n",l_PacketNames[packet->mPacketType],name,((cCalcPacket*)packet)->GetTotal());
        break;
      case cRulePacket::CLASS_HIT:
        {
          Log("Exit %s: %s; Total %d\n",l_PacketNames[packet->mPacketType],name,((cCalcPacket*)packet)->GetTotal());
        }

        break;
      case cRulePacket::CLASS_DAMAGE:
        {
          Log("Exit %s: %s; Total: %d\n",l_PacketNames[packet->mPacketType],name,((cCalcPacket*)packet)->GetTotal());
        }
        break;
      case cRulePacket::CLASS_FLAG:
        {
        }
        break;
      default:
        SyAssertf(0,"Unknown packet type");
      }
    }
    if (l_logHiddenLevel == l_logIndent)
    {
      l_logHiddenLevel = 1000;
    }
  }
}

//------------------------------------------------------ cRuleSys
cRuleSys::cRuleSys()
{
}

cRuleSys::~cRuleSys()
{
  for (int i=mRules.Begin(); i!=mRules.End(); i=mRules.Next(i))
  {
    delete mRules(i);
  }
  mRules.Clear();
}

void cRuleSys::Init()
{
  Add(SyNew cRule_Invisible());
  Add(SyNew cRule_Invulnerable());
  Add(SyNew cRule_DamageDone());
  Add(SyNew cRule_DamageTaken());
  Add(SyNew cRule_Stunned());
  Add(SyNew cRule_Sleeping());
  Add(SyNew cRule_Confused());
  Add(SyNew cRule_Charmed());
  Add(SyNew cRule_Frozen());
  Add(SyNew cRule_RandomTeleport());
  Add(SyNew cRule_DirectionalTeleport());
  Add(SyNew cRule_Resurrecting());
  Add(SyNew cRule_DelayedSpell());
  Add(SyNew cRule_DelayedAttackSpell());
  Add(SyNew cRule_DelayedDefendSpell());
  Add(SyNew cRule_DelayedDeathSpell());
  Add(SyNew cRule_DelayedDeath());
  Add(SyNew cRule_HealthDrain());
  Add(SyNew cRule_ManaDrain());
  Add(SyNew cRule_DamageShieldPercent());
  Add(SyNew cRule_ManaShieldPercent());
  Add(SyNew cRule_TargetMe());
  Add(SyNew cRule_Silence());
  Add(SyNew cRule_Bind());
  Add(SyNew cRule_Fear());
  Add(SyNew cRule_Immunity());
  Add(SyNew cRule_KnockdownImmunity());
  Add(SyNew cRule_KnockbackImmunity());
  Add(SyNew cRule_VortexImmunity());
  Add(SyNew cRule_KillImmunity());
  Add(SyNew cRule_Illusion());
  Add(SyNew cRule_AlterAttackSpeed());
  Add(SyNew cRule_AlterMovementSpeed());
  Add(SyNew cRule_AlterAttackPower());
  Add(SyNew cRule_AlterSpellPower());
  Add(SyNew cRule_AlterBlockPower());
  Add(SyNew cRule_AlterCriticalChance());
  Add(SyNew cRule_AlterDodgeChance());
  Add(SyNew cRule_AlterDefense());
  Add(SyNew cRule_AlterSize());
  Add(SyNew cRule_AlterHealthRegen());
  Add(SyNew cRule_AlterManaRegen());
  Add(SyNew cRule_AlterExperienceGain());
  Add(SyNew cRule_AlterHealthAbsorb());
  Add(SyNew cRule_AlterManaAbsorb());
  Add(SyNew cRule_AlterEssenceAbsorb());
  Add(SyNew cRule_AlterHealthDropChance());
  Add(SyNew cRule_AlterManaDropChance());
  Add(SyNew cRule_AlterEssenceDropChance());
  Add(SyNew cRule_AlterItemDropChance());
  Add(SyNew cRule_DamageOverTime());
  Add(SyNew cRule_OnFire());
  Add(SyNew cRule_ManaCostOverTime());
}

cRule_Condition* cRuleSys::CreateCondition(const char* conditionName,
                                           float duration,
                                           int param1, int param2)
{
  tGameID conditionID = SyHashResourceID(conditionName);

  int index = mRules.Find(conditionID);
  
  SyAssertf(index != mRules.End(), "Condition name '%s' not found in rulesys", conditionName);

  if (index != mRules.End())
  {
    cRule_Condition* pRule = mRules(index);
    SyAssertf(pRule!=NULL, "Matching rule for condition name '%s is bad'", conditionName);

    if (pRule)
    {
      cRule_Condition* pNewRule = pRule->Clone();
      SyAssertf(pNewRule!=NULL && strcmp(pNewRule->GetName(), pRule->GetName())==0, "cRuleSys bad rule create for '%s' (no overloaded Clone() member?)", conditionName);

      if (pNewRule)
      {
        SyAssertf(pNewRule->GetName()!=NULL && ((int)strlen(pNewRule->GetName())) <= cRule::MAX_NAME_LEN, "Rule %s has name that's too long (max %d)", pNewRule->GetName(), cRule::MAX_NAME_LEN);
        pNewRule->SetTitan(Titan::Get());
        pNewRule->SetDuration(duration);
        pNewRule->Init(param1, param2);
        return pNewRule;
      }
    }
  }

  return NULL;
}

void cRuleSys::Add(cRule_Condition* pRule)
{
  SyAssert(pRule!=NULL);

  if (!pRule)
  {
    return;
  }

  tGameID id = SyHashResourceID(pRule->GetName());
  mRules.Insert(id, pRule);
}



// EOF
