/******************************************************************
  
  Module:  rule_condition.h
  
  Author: Sean Craig
  
  Description:  Rule event handlers with durations.  For example,
  the Poison condition does damage every update, and removes 
  itself after a duration.

  
  Copyright 2005 Sony Online Entertainment.  All rights reserved.
  
*******************************************************************/

#ifndef RULE_CONDITION_H
#define RULE_CONDITION_H

//-------------------------------------------------------- Includes
#include "rule.h"
//-------------------------------------------------------- Typedefs
//---------------------------------------------- Class Declarations

//---------------------------------------------- class cRule_Condition
class cRule_Condition : public cRule
{
public:
  cRule_Condition();
  virtual ~cRule_Condition(){};

  virtual void                Init(int param1, int param2);
  virtual void                Update(float time); 
  virtual cRule_Condition*    Clone() { return SyNew cRule_Condition(*this); }

  virtual bool              Combine(cRule *other); 
  virtual bool              Block(cRule *other); 

  virtual int               GetStrength()      { return 0; }
  virtual void              SetStrength(int s) {}

  void SetDuration(float t);

  virtual void    SetEffect(tGameID effectID)  { mEffectID = effectID; }

protected:
  cRule_Condition(const cRule_Condition& rule); // protect so everybody else must use clone

  virtual bool OnAdded(tGameObjectID target);
  virtual bool OnRemoved(tGameObjectID target);

  float           mRemainingTime;
  tGameID         mEffectID;
};

//---------------------------------------------- class cRule_AlterStat
class cRule_AlterStat : public cRule_Condition
{
public:
  cRule_AlterStat() : mStrength(0) {}

  virtual int   GetStrength() { return mStrength; }
  virtual void  SetStrength(int s) { mStrength = s; }

  virtual void Init(int param1, int param2);
//  virtual bool  Combine(cRule *other);

protected:
  float GetStrengthNormalized() { return ((float)mStrength)*0.01f; }

  int mStrength; // percent from -100 to 100, 
};


//---------------------------------------------- class cRule_Invisible
class cRule_Invisible : public cRule_Condition
{
public:
  cRule_Invisible() : mOwnerID(ID_NONE), mbWasAttacking(false) {}
  virtual const char * GetName() {return "Invisible";}
  virtual cRule_Condition* Clone() { return SyNew cRule_Invisible(*this); }

  virtual void Update(float time); 

protected:
  virtual void QueryFlag(cFlagPacket *packet);
  virtual bool OnAdded(tGameObjectID target);
  virtual bool OnRemoved(tGameObjectID target);

  tGameObjectID mOwnerID;
  bool          mbWasAttacking;
};

//---------------------------------------------- class cRule_Stunned
class cRule_Stunned : public cRule_Condition
{
public:
  virtual const char*     GetName() {return "Stunned";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Stunned(*this); }

protected:
  virtual void QueryFlag(cFlagPacket *packet);
  virtual void CalcAttackSpeed(cCalcPacket *packet); 
  virtual void CalcMovementSpeed(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_Confused
class cRule_Confused : public cRule_Condition
{
public:
  cRule_Confused() : cRule_Condition(), mOwnerID(ID_NONE) {}
  virtual const char*     GetName() { return "Confused"; };
  virtual cRule_Condition* Clone() { return SyNew cRule_Confused(*this); }

  virtual void Update(float time); 

protected:
  virtual bool OnAdded(tGameObjectID target);
  virtual bool OnRemoved(tGameObjectID target);
  virtual void QueryFlag(cFlagPacket *packet);

  tGameObjectID PickRandomTarget();

  tGameObjectID mOwnerID;
};

//---------------------------------------------- class cRule_Charmed
class cRule_Charmed : public cRule_Condition
{
public:
  virtual const char*     GetName() { return "Charmed"; };
  virtual cRule_Condition* Clone() { return SyNew cRule_Charmed(*this); }

protected:
  virtual bool OnAdded(tGameObjectID target);
  virtual bool OnRemoved(tGameObjectID target);
  virtual void QueryFlag(cFlagPacket *packet);
};

//---------------------------------------------- class cRule_Sleeping
class cRule_Sleeping : public cRule_Condition
{
public:
  virtual const char*     GetName() {return "Sleeping";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Sleeping(*this); }

protected:
  virtual void QueryFlag(cFlagPacket *packet);
  virtual void OnApplyDamage(cDamagePacket *packet); 
  virtual void CalcAttackSpeed(cCalcPacket *packet); 
  virtual void CalcMovementSpeed(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_Frozen
class cRule_Frozen : public cRule_Condition
{
public:
  virtual const char*     GetName() {return "Frozen";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Frozen(*this); }

  virtual bool Combine(cRule *other); 

protected:
  virtual void QueryFlag(cFlagPacket* packet);
  virtual void OnApplyDamage(cDamagePacket *packet); 
  virtual void CalcAttackSpeed(cCalcPacket *packet); 
  virtual void CalcMovementSpeed(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_Silenced
class cRule_Silence : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Silenced";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Silence(*this); }

protected:
  virtual void QueryFlag(cFlagPacket *packet);
};

//---------------------------------------------- class cRule_Bound
class cRule_Bind : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Bound";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Bind(*this); }

protected:
  virtual void CalcAttackSpeed(cCalcPacket *packet); 
  virtual void QueryFlag(cFlagPacket *packet);
};

//---------------------------------------------- class cRule_Fear
class cRule_Fear : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Fear";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Fear(*this); }

protected:
  virtual bool OnAdded(tGameObjectID target);
};

//---------------------------------------------- class cRule_Immunity
class cRule_Immunity : public cRule_Condition
{
public:
  cRule_Immunity() : mRuleNameID(ID_NONE) {}
  virtual const char * GetName(){return "Immunity";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Immunity(*this); }

  virtual void              Init(int param1, int param2);
  virtual bool              Combine(cRule *other); 
  virtual bool              Block(cRule *other); 

protected:
  tGameID mRuleNameID;
};

//---------------------------------------------- class cRule_KnockdownImmunity
class cRule_KnockdownImmunity : public cRule_Condition
{
public:
  cRule_KnockdownImmunity() {}
  virtual const char * GetName(){return "Knockdown Immunity";};
  virtual cRule_Condition* Clone() { return SyNew cRule_KnockdownImmunity(*this); }

protected:
  virtual void QueryFlag(cFlagPacket *packet);
};

//---------------------------------------------- class cRule_KnockbackImmunity
class cRule_KnockbackImmunity : public cRule_Condition
{
public:
  cRule_KnockbackImmunity() {}
  virtual const char * GetName(){return "Knockback Immunity";};
  virtual cRule_Condition* Clone() { return SyNew cRule_KnockbackImmunity(*this); }

protected:
  virtual void QueryFlag(cFlagPacket *packet);
};

//---------------------------------------------- class cRule_VortexImmunity
class cRule_VortexImmunity : public cRule_Condition
{
public:
  cRule_VortexImmunity() {}
  virtual const char * GetName(){return "Vortex Immunity";};
  virtual cRule_Condition* Clone() { return SyNew cRule_VortexImmunity(*this); }

protected:
  virtual void QueryFlag(cFlagPacket *packet);
};

//---------------------------------------------- class cRule_KillImmunity
class cRule_KillImmunity : public cRule_Condition
{
public:
  cRule_KillImmunity() {}
  virtual const char * GetName(){return "Kill Immunity";};
  virtual cRule_Condition* Clone() { return SyNew cRule_KillImmunity(*this); }

protected:
  virtual void QueryFlag(cFlagPacket *packet);
};

//---------------------------------------------- class cRule_Illusion
class cRule_Illusion : public cRule_Condition
{
public:
  cRule_Illusion() : mOldMasterID(ID_NONE) {}
  virtual const char * GetName(){return "Illusion";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Illusion(*this); }

protected:
  virtual bool OnAdded(tGameObjectID target);
  virtual bool OnRemoved(tGameObjectID target);

  tGameID mOldMasterID;
};

//---------------------------------------------- class cRule_Invulnerable
class cRule_Invulnerable : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Invulnerable";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Invulnerable(*this); }
protected:
  virtual void OnApplyDamage(cDamagePacket *packet);
  virtual void QueryFlag(cFlagPacket *packet);
};

//---------------------------------------------- class cRule_RandomTeleport
class cRule_RandomTeleport : public cRule_Condition
{
public:
  cRule_RandomTeleport() : mOwnerID(ID_NONE), mStart(0.0f, 0.0f, 0.0f), mHeading(0.0f), mDist(5.0f), mNumTests(0) {}
  virtual const char * GetName(){return "Random Teleport";};
  virtual cRule_Condition* Clone() { return SyNew cRule_RandomTeleport(*this); }

  virtual void Init(int param1, int param2);
  virtual void Update(float time); 
protected:
  virtual void QueryFlag(cFlagPacket *packet);
  virtual bool OnAdded(tGameObjectID target);

  tGameObjectID mOwnerID;
  SyVect3       mStart;
  float         mHeading;
  float         mDist;
  int           mNumTests;
};

//---------------------------------------------- class cRule_DirectionalTeleport
class cRule_DirectionalTeleport : public cRule_RandomTeleport
{
public:
  cRule_DirectionalTeleport() : cRule_RandomTeleport() {}
  virtual const char * GetName(){return "Directional Teleport";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DirectionalTeleport(*this); }

  virtual void Init(int param1, int param2);
};

//---------------------------------------------- class cRule_Resurrecting
class cRule_Resurrecting : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Resurrecting";};
  virtual cRule_Condition* Clone() { return SyNew cRule_Resurrecting(*this); }

protected:
  virtual bool OnRemoved(tGameObjectID target);
};

//---------------------------------------------- class cRule_DelayedSpell
class cRule_DelayedSpell : public cRule_Condition
{
public:
  cRule_DelayedSpell() : mOwnerID(ID_NONE), mDelayedSpellID(ID_NONE), mSpellTargetPos(0.0f, 0.0f, 0.0f) {}
  virtual const char * GetName(){return "Delayed Spell";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DelayedSpell(*this); }

  virtual void Init(int param1, int param2);
  virtual void Update(float time); 

  virtual bool Combine(cRule *other); 
  virtual bool Block(cRule *other); 

protected:
  virtual bool OnAdded(tGameObjectID target);

  tGameObjectID mOwnerID;
  tGameID       mDelayedSpellID;
  SyVect3       mSpellTargetPos;
};

//---------------------------------------------- class cRule_DelayedAttackSpell
class cRule_DelayedAttackSpell : public cRule_DelayedSpell
{
public:
  virtual const char * GetName(){return "Delayed Attack Spell";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DelayedAttackSpell(*this); }

  virtual void Update(float time); 

protected:
  virtual void    OnHit(cHitPacket *packet);
};

//---------------------------------------------- class cRule_DelayedDefendSpell
class cRule_DelayedDefendSpell : public cRule_DelayedSpell
{
public:
  virtual const char * GetName(){return "Delayed Defend Spell";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DelayedDefendSpell(*this); }

  virtual void Update(float time); 

protected:
  virtual void    OnApplyDamage(cDamagePacket *packet);
};

//---------------------------------------------- class cRule_DelayedDeathSpell
class cRule_DelayedDeathSpell : public cRule_DelayedSpell
{
public:
  virtual const char * GetName(){return "Delayed Death Spell";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DelayedDeathSpell(*this); }

  virtual void Update(float time); 
protected:
};

//---------------------------------------------- class cRule_DelayedDeath
class cRule_DelayedDeath : public cRule_Condition
{
public:
  cRule_DelayedDeath() : mOwnerID(ID_NONE) {}
  virtual const char * GetName(){return "Delayed Death";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DelayedDeath(*this); }

  virtual void Update(float time); 

protected:
  virtual bool OnAdded(tGameObjectID target);

  tGameObjectID mOwnerID;
};

//---------------------------------------------- class cRule_HealthDrain
class cRule_HealthDrain : public cRule_Condition
{
public:
  cRule_HealthDrain() : mDrain(0), mOwnerID(ID_NONE) {}
  virtual const char * GetName(){return "Health Drain";};
  virtual cRule_Condition* Clone() { return SyNew cRule_HealthDrain(*this); }
  virtual void    Update(float time); 
protected:
  virtual bool    OnAdded(tGameObjectID target);
  virtual void    OnHit(cHitPacket *packet);

  int mDrain;
  tGameObjectID mOwnerID;
};

//---------------------------------------------- class cRule_ManaDrain
class cRule_ManaDrain : public cRule_Condition
{
public:
  cRule_ManaDrain() : mDrain(0), mOwnerID(ID_NONE) {}
  virtual const char * GetName(){return "Mana Drain";};
  virtual cRule_Condition* Clone() { return SyNew cRule_ManaDrain(*this); }
  virtual void    Update(float time); 
protected:
  virtual bool    OnAdded(tGameObjectID target);
  virtual void    OnHit(cHitPacket *packet);

  int mDrain;
  tGameObjectID mOwnerID;
};

//---------------------------------------------- class cRule_DamageShieldPercent
class cRule_DamageShieldPercent : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Damage Shield Percent";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DamageShieldPercent(*this); }
  virtual void Init(int param1, int param2);
protected:
  virtual void    OnApplyDamage(cDamagePacket *packet);
  int mPercent;
};

//---------------------------------------------- class cRule_ManaShield
class cRule_ManaShieldPercent : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Mana Shield Percent";};
  virtual cRule_Condition* Clone() { return SyNew cRule_ManaShieldPercent(*this); }
  virtual void Init(int param1, int param2);
protected:
  virtual void    OnApplyDamage(cDamagePacket *packet);
  int mPercent;
};

//---------------------------------------------- class cRule_TargetMe
class cRule_TargetMe : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Target Me";};
  virtual cRule_Condition* Clone() { return SyNew cRule_TargetMe(*this); }
protected:
  virtual void QueryFlag(cFlagPacket* packet);
};

//---------------------------------------------- class cRule_DamageTaken
class cRule_DamageTaken : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Damage Taken";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DamageTaken(*this); }

  virtual void Init(int param1, int param2);

protected:
  virtual void CalcDamageReduced(cDamagePacket *packet);

  float mDmgMult;
};

//---------------------------------------------- class cRule_DamageDone
class cRule_DamageDone : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Damage Done";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DamageDone(*this); }

  virtual void Init(int param1, int param2);

protected:
  virtual void CalcDamageRaw(cDamagePacket *packet);

  float mDmgMult;
};


//---------------------------------------------- class cRule_AlterAttackSpeed
class cRule_AlterAttackSpeed : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Attack Speed";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterAttackSpeed(*this); }

protected:
  virtual void CalcAttackSpeed(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_AlterMovementSpeed
class cRule_AlterMovementSpeed : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Movement Speed";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterMovementSpeed(*this); }

protected:
  virtual void CalcMovementSpeed(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_AlterAttackPower
class cRule_AlterAttackPower : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Attack Power";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterAttackPower(*this); }

protected:
  virtual void    CalcDamageInitial(cDamagePacket *packet);
};

//---------------------------------------------- class cRule_AlterSpellPower
class cRule_AlterSpellPower : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Spell Power";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterSpellPower(*this); }

protected:
  virtual void    CalcSpellPower(cCalcPacket *packet);
};

//---------------------------------------------- class cRule_AlterBlockPower
class cRule_AlterBlockPower : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Block Power";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterBlockPower(*this); }

protected:
  virtual void    CalcBlockMax(cCalcPacket *packet);
  virtual void    CalcBlockRegen(cCalcPacket *packet);
  virtual void    CalcStrikethrough(cDamagePacket *packet);
};

//---------------------------------------------- class cRule_AlterCriticalChance
class cRule_AlterCriticalChance : public cRule_AlterStat
{
public:
  cRule_AlterCriticalChance() : cRule_AlterStat(), mbRanged(false) {}
  virtual const char * GetName() {return "Altered Critical Chance";}
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterCriticalChance(*this); }

  virtual void Init(int param1, int param2);

protected:
  virtual void CalcCriticalChance(cDamagePacket *packet);

  bool mbRanged;
};

//---------------------------------------------- class cRule_AlterDodgeChance
class cRule_AlterDodgeChance : public cRule_AlterStat
{
public:
  cRule_AlterDodgeChance() : cRule_AlterStat(), mbRanged(false) {}
  virtual const char * GetName() {return "Altered Dodge Chance";}
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterDodgeChance(*this); }

  virtual void Init(int param1, int param2);

protected:
  virtual void CalcDodgeChance(cDamagePacket *packet);

  bool mbRanged;
};

//---------------------------------------------- class cRule_AlterNormalResist
class cRule_AlterDefense : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Defense";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterDefense(*this); }

  virtual void Init(int param1, int param2);
protected:
  virtual void CalcDefense(cDamagePacket *packet);

  eDamageType mDmgType;
};

//---------------------------------------------- class cRule_AlterSize
class cRule_AlterSize : public cRule_AlterStat
{
public:
  cRule_AlterSize();
  virtual const char * GetName(){return "Altered Size";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterSize(*this); }

  virtual void    SetDisabled(); // disable after a short period time
  virtual void    Update(float time); 
  virtual bool    Combine(cRule *other); 

protected:
  static const float ANIMATE_SIZE_TIME;

  virtual bool OnAdded(tGameObjectID target);

  tGameObjectID mOwnerID;
  float mTimer;
  bool mbStartDisable;
};

//---------------------------------------------- class cRule_AlterHealthRegen
class cRule_AlterHealthRegen : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Health Regen";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterHealthRegen(*this); }

protected:
  virtual void CalcHealthRegen(cCalcPacket *packet);
};

//---------------------------------------------- class cRule_AlterManaRegen
class cRule_AlterManaRegen : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Mana Regen";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterManaRegen(*this); }

protected:
  virtual void CalcManaRegen(cCalcPacket *packet);
};

//---------------------------------------------- class cRule_AlterExperienceGain
class cRule_AlterExperienceGain : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Experience Gain";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterExperienceGain(*this); }

protected:
  virtual void CalcExperienceGain(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_AlterHealthAbsorb
class cRule_AlterHealthAbsorb : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Health Absorb";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterHealthAbsorb(*this); }

protected:
  virtual void CalcHealthAbsorb(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_AlterManaAbsorb
class cRule_AlterManaAbsorb : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Mana Absorb";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterManaAbsorb(*this); }

protected:
  virtual void CalcManaAbsorb(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_AlterEssenceAbsorb
class cRule_AlterEssenceAbsorb : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Essence Absorb";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterEssenceAbsorb(*this); }

protected:
  virtual void CalcEssenceAbsorb(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_AlterHealthDropChance
class cRule_AlterHealthDropChance : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Health Drop Chance";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterHealthDropChance(*this); }

protected:
  virtual void CalcHealthDropChance(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_AlterManaDropChance
class cRule_AlterManaDropChance : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Mana Drop Chance";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterManaDropChance(*this); }

protected:
  virtual void CalcManaDropChance(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_AlterEssenceDropChance
class cRule_AlterEssenceDropChance : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Essence Drop Chance";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterEssenceDropChance(*this); }

protected:
  virtual void CalcEssenceDropChance(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_AlterItemDropChance
class cRule_AlterItemDropChance : public cRule_AlterStat
{
public:
  virtual const char * GetName(){return "Altered Item Drop Chance";};
  virtual cRule_Condition* Clone() { return SyNew cRule_AlterItemDropChance(*this); }

protected:
  virtual void CalcItemDropChance(cCalcPacket *packet); 
};

//---------------------------------------------- class cRule_DamageOverTime
class cRule_DamageOverTime : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Damage Over Time";};
  virtual cRule_Condition* Clone() { return SyNew cRule_DamageOverTime(*this); }

  virtual void Update(float time);
  virtual void Init(int param1, int param2);
protected:
  virtual bool    OnAdded(tGameObjectID target);

  tGameObjectID mOwnerID;
  int mDmgPerSec;
  eDamageType mDmgType;
  float mTimer;
};

//---------------------------------------------- class cRule_OnFire
class cRule_OnFire : public cRule_DamageOverTime
{
public:
  cRule_OnFire();

  virtual const char * GetName(){return "On Fire";};
  virtual cRule_Condition* Clone() { return SyNew cRule_OnFire(*this); }

  virtual void Init(int param1, int param2);
  virtual void Update(float time);
  virtual bool Combine(cRule *other); 

protected:
  virtual void QueryFlag(cFlagPacket *packet);

  tGameID mAreaEffectID;
  float mLifetime;
  bool mbOwnerDied;

};

//---------------------------------------------- class cRule_ManaCostOverTime
class cRule_ManaCostOverTime : public cRule_Condition
{
public:
  virtual const char * GetName(){return "Mana Cost Over Time";};
  virtual cRule_Condition* Clone() { return SyNew cRule_ManaCostOverTime(*this); }

  virtual void Update(float time);
  virtual void Init(int param1, int param2);
protected:
  virtual bool    OnAdded(tGameObjectID target);

  tGameObjectID mOwnerID;
  int mCostPerSec;
  float mTimer;
};
//------------------------------------------- Function Declarations
//--------------------------------------------------------- Globals
//------------------------------------------------ Inline Functions

#endif
