/******************************************************************
  
  Module:  script.cpp
  
  Author: Sean Craig
  
  Copyright 2005 Sony Online Entertainment.  All rights reserved.
  
*******************************************************************/

//-------------------------------------------------------- Includes
#include "script.h"
#include "animcontroller.h"
#include "registry.h"
#include "graphic.h"
#include "Titan.h"
#include "SyScene.h"
#include "levelobj.h"
//---------------------------------------------- Class Declarations
//----------------------------------------- Functions Declarations
//------------------------------------ Member Functions Definitions

///////////////////////////////////////////////////////////////////////////// cScriptFrame 

cScriptFrame::cScriptFrame()
: mSelf(ID_NONE),
  mActor(ID_NONE),
  mState(SS_ON),
  mEvent(SE_ACTIVATE),
  mOwner(NULL),
  mResponseIndex(0)
{
};

///////////////////////////////////////////////////////////////////////////// cScript

cScript::cScript() 
  : mOwner(NULL),
    mState(SS_ON)
{
  InitPropObject( SCRIPT_CLASSID );
  for (int ii=0;ii<NUM_SCRIPT_STATES;++ii)
  {
    mStates[ii] = NULL;
  }
  mOwner = NULL;
}

cScript::~cScript()
{
  for (int ii=0;ii<NUM_SCRIPT_STATES;++ii)
  {
    delete mStates[ii];
    mStates[ii] = NULL;
  }
}

int                  
cScript::InitPropClass()
{
  SyPropClassID ClassID = SCRIPT_CLASSID;

/* Add the class */

  AddClass( ClassID, 
            "cScript", 
            Creator, 
            ClassID, 
            0 ); 


  for (int ii=0;ii<NUM_SCRIPT_STATES;++ii)
  {
    char propname[256];
    sprintf(propname,"State_%d",ii);
    AddSubObjectPtrProperty(ClassID,ii,SyMemberOffset(cScript,mStates[ii]),propname);
  }

  return 0;
}

SyPropObject*        
cScript::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScript();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

cScriptBlock*
cScript::GetState( eScriptState state )
{
  SyAssert( state >= 0 && state < NUM_SCRIPT_STATES );
  return( mStates[state] );
}

void 
cScript::Clear()
{
  for( int i = 0; i < NUM_SCRIPT_STATES; i++ )
  {
    delete mStates[i];
    mStates[i] = SyNew cScriptBlock();
  }
}

void
cScript::SetOwner( cGameObject* pOwner )
{
  mOwner = pOwner;
}

void
cScript::Update( float time )
{
}

void 
cScript::Run(cGameObject *actor,eOldScriptEventType event)
{
  if (mOwner->GetTitan()->GetRegistry()->GetLevelObject()->IsRemote())
  {
    return;
  }

  cScriptFrame frame;
  if (actor != NULL)
  {
    frame.mActor = actor->GetID();
  }

  frame.mEvent = event;
  frame.mResponseIndex = 0;
  frame.mSelf = mOwner->GetID();
  frame.mState = mState;
  frame.mOwner = mOwner;
  Process(&frame);
}

eScriptResult
cScript::Process( cScriptFrame* frame )
{
  if (mStates[frame->mState] == NULL)
  {
    return SR_EXIT;
  }
  return mStates[frame->mState]->Process(frame);
}

///////////////////////////////////////////////////////////////////////////// cScriptBlock

cScriptBlock::cScriptBlock()
{
  InitPropObject( SCRIPTBLOCK_CLASSID );
}

cScriptBlock::~cScriptBlock()
{
  for (int ii=mHandlers.Begin();ii!=mHandlers.End();ii=mHandlers.Next(ii))
  {
    delete mHandlers(ii);
  }
  mHandlers.Clear();
}

  /* Property Class Support */
int                  
cScriptBlock::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTBLOCK_CLASSID;

/* Add the class */

  AddClass( ClassID, 
            "cScriptBlock", 
            Creator, 
            ClassID, 
            0 ); 

  AddSubObjectPtrMapProperty<int32,cScriptEventHandler *>(ClassID,
                                                          0x0000,
                                                          SyMemberOffset(cScriptBlock,mHandlers),
                                                          "mHandlers",
                                                           SYPROPTYPE_INT32);

  return 0;
}

SyPropObject*        
cScriptBlock::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptBlock();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

eScriptResult 
cScriptBlock::Process(cScriptFrame *frame)
{
  int index = mHandlers.Find(frame->mEvent);

  if (index != mHandlers.End())
  {
    return mHandlers(index)->Process(frame);
  }
  return SR_EXIT;
}

///////////////////////////////////////////////////////////////////////////// cScriptEventHandler

cScriptEventHandler::cScriptEventHandler()
{
  InitPropObject( SCRIPTEVENTHANDLER_CLASSID );
}

cScriptEventHandler::~cScriptEventHandler()
{
  for (int ii=mResponses.Begin();ii!=mResponses.End();ii=mResponses.Next(ii))
  {
    delete mResponses(ii);
  }
  mResponses.Clear();
}

int                  
cScriptEventHandler::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTEVENTHANDLER_CLASSID;

/* Add the class */

  AddClass( ClassID, 
            "cScriptEventHandler", 
            Creator, 
            ClassID, 
            0 ); 

  AddSubObjectPtrVectorProperty<cScriptResponse>( 
              ClassID, 
              0x0000,
              SyMemberOffset(cScriptEventHandler,mResponses),
              "mResponses");
  return 0;
}

SyPropObject*        
cScriptEventHandler::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptEventHandler();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}


eScriptResult 
cScriptEventHandler::Process(cScriptFrame *frame)
{
  while(1)
  {
    if (frame->mResponseIndex >= mResponses.Size())
    {
      return SR_EXIT;
    }

    switch (mResponses(frame->mResponseIndex)->Process(frame))
    {
      case SR_EXIT:
        return SR_EXIT;
        break;
      case SR_NEXT:
        break;
      case SR_SKIP:
      {
        frame->mResponseIndex++;
        Skip(frame);
      }
    }
    frame->mResponseIndex++;
  }
  return SR_NEXT;
}


eScriptResult
cScriptEventHandler::Skip(cScriptFrame *frame)
{
  int depth = 1;
  while (frame->mResponseIndex  < mResponses.Size())
  {
    switch (mResponses(frame->mResponseIndex)->GetType())
    {
      case SRT_STATEMENT:
        break;
      case SRT_CHECK:
        ++depth; // nested if
        break;
      case SRT_ENDIF:
        --depth;
        if (depth==0)
        {
          return SR_NEXT;
        }
        break;
      case SRT_ELSE:
        if (depth == 1)
        {
          // we've skipped the block of code we care about, go ahead and execute this block.
          return SR_NEXT;
        }
        break;
    }
    frame->mResponseIndex++;
  }
  return SR_EXIT;
}



///////////////////////////////////////////////////////////////////////////// cScriptResponse

cScriptResponse::cScriptResponse()
{
  InitPropObject( SCRIPTRESPONSE_CLASSID );
}

int                  
cScriptResponse::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_CLASSID;

/* Add the class */

  AddClass( ClassID, 
            "cScriptResponse", 
            NULL,               // CRO: abstract types have no creator
            ClassID, 
            0 ); 
  return 0;
}

SyPropObject*        
cScriptResponse::Creator()
{

  SyAssertf(0,"Trying to create an abstract base class");
  return(NULL);
}

///////////////////////////////////////////////////////////////////////////// cScriptResponseCheckState

cScriptResponseCheckState::cScriptResponseCheckState()
{
  InitPropObject( SCRIPTRESPONSE_CHECKSTATE_CLASSID );

  mSpecified = 0;
  mState = SS_ON;
  mTarget = ST_SELF;
}

int                  
cScriptResponseCheckState::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_CHECKSTATE_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseCheckState", 
               Creator, 
               ClassID, 
               0 ); 

  SyPropEnum *propEnum;
  AddEnumProperty(ClassID,0x1000,SyMemberOffset(cScriptResponseCheckState,mTarget),"mTarget",&propEnum);
  propEnum->Add(ST_SELF,"SELF");
  propEnum->Add(ST_ACTOR,"ACTOR");
  propEnum->Add(ST_SPECIFIED,"SPECIFIED");

  AddInt32Property(ClassID,0x1001,SyMemberOffset(cScriptResponseCheckState,mSpecified),"mSpecified");

  AddEnumProperty(ClassID,0x1002,SyMemberOffset(cScriptResponseCheckState,mState),"mState",&propEnum);
  propEnum->Add(SS_ON,"ON");
  propEnum->Add(SS_OFF,"OFF");
  propEnum->Add(SS_DEAD,"DEAD");
  return 0;
}

SyPropObject*        
cScriptResponseCheckState::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseCheckState();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

eScriptResult 
cScriptResponseCheckState::Process(cScriptFrame *frame)
{
  // if target.state ==  mState return SR_NEXT, otherwise return SR_SKIP
  if( frame->mState == mState )
  {
    return SR_NEXT;
  }
  else
  {
    return SR_SKIP;
  }
};

///////////////////////////////////////////////////////////////////////////// cScriptResponseEndif

cScriptResponseEndif::cScriptResponseEndif()
{
  InitPropObject( SCRIPTRESPONSE_ENDIF_CLASSID );
}


int                  
cScriptResponseEndif::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_ENDIF_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseeEndif", 
               Creator, 
               ClassID, 
               0 ); 
  return 0;
}

SyPropObject*        
cScriptResponseEndif::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseEndif();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

///////////////////////////////////////////////////////////////////////////// cScriptResponseElse

cScriptResponseElse::cScriptResponseElse()
{
  InitPropObject( SCRIPTRESPONSE_ELSE_CLASSID );
}

int                  
cScriptResponseElse::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_ELSE_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseElse", 
               Creator, 
               ClassID, 
               0 ); 
  return 0;
}

SyPropObject*        
cScriptResponseElse::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseElse();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}


///////////////////////////////////////////////////////////////////////////// cScriptResponseChangeState

cScriptResponseChangeState::cScriptResponseChangeState()
{
  InitPropObject( SCRIPTRESPONSE_CHANGESTATE_CLASSID );

  mSpecified = 0;
  mState = SS_ON;
  mTarget = ST_SELF;
}

int                  
cScriptResponseChangeState::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_CHANGESTATE_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseChangeState", 
               Creator, 
               ClassID, 
               0 ); 
  SyPropEnum *propEnum;
  AddEnumProperty(ClassID,0x1000,SyMemberOffset(cScriptResponseChangeState,mTarget),"mTarget",&propEnum);
  propEnum->Add(ST_SELF,"SELF");
  propEnum->Add(ST_ACTOR,"ACTOR");
  propEnum->Add(ST_SPECIFIED,"SPECIFIED");

  AddInt32Property(ClassID,0x1001,SyMemberOffset(cScriptResponseChangeState,mSpecified),"mSpecified");

  AddEnumProperty(ClassID,0x1002,SyMemberOffset(cScriptResponseChangeState,mState),"mState",&propEnum);
  propEnum->Add(SS_ON,"ON");
  propEnum->Add(SS_OFF,"OFF");
  propEnum->Add(SS_DEAD,"DEAD");
  return 0;
}

SyPropObject*        
cScriptResponseChangeState::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseChangeState();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

eScriptResult
cScriptResponseChangeState::Process( cScriptFrame* frame )
{
  frame->mOwner->GetScript()->SetState( mState );
  return SR_NEXT;
}

///////////////////////////////////////////////////////////////////////////// cScriptResponsePlayMusic

cScriptResponsePlayMusic::cScriptResponsePlayMusic()
{
  InitPropObject( SCRIPTRESPONSE_PLAYMUSIC_CLASSID );
}

int                  
cScriptResponsePlayMusic::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_PLAYMUSIC_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponsePlayMusic", 
               Creator, 
               ClassID, 
               0 ); 

  AddStringProperty(ClassID,0x1000,SyMemberOffset(cScriptResponsePlayMusic,mMusicName),"mMusicName");
  return 0;

}

SyPropObject*        
cScriptResponsePlayMusic::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponsePlayMusic();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}


///////////////////////////////////////////////////////////////////////////// cScriptResponsePlaySound

cScriptResponsePlaySound::cScriptResponsePlaySound()
{
  InitPropObject( SCRIPTRESPONSE_PLAYSOUND_CLASSID );

  mSpacialized = true;
  mTarget = ST_SELF;
  mVolume = 1.0f;
  mTargetLocation = 0;
}

int                  
cScriptResponsePlaySound::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_PLAYSOUND_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponsePlaySound", 
               Creator, 
               ClassID, 
               0 ); 
  AddStringProperty(ClassID,0x1000,SyMemberOffset(cScriptResponsePlaySound,mSoundName),"mSoundName");
  AddBoolProperty(ClassID,0x1001,SyMemberOffset(cScriptResponsePlaySound,mSpacialized),"mSpacialized");
  AddFloat32Property(ClassID,0x1002,SyMemberOffset(cScriptResponsePlaySound,mVolume),"mVolume");
  SyPropEnum *propEnum;
  AddEnumProperty(ClassID,0x1003,SyMemberOffset(cScriptResponsePlaySound,mTarget),"mTarget",&propEnum);
  propEnum->Add(ST_SELF,"SELF");
  propEnum->Add(ST_ACTOR,"ACTOR");
  propEnum->Add(ST_SPECIFIED,"SPECIFIED");

  AddInt32Property(ClassID,0x1004,SyMemberOffset(cScriptResponsePlaySound,mTargetLocation),"mTargetLocation");
  return 0;

}

SyPropObject*        
cScriptResponsePlaySound::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponsePlaySound();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

eScriptResult
cScriptResponsePlaySound::Process( cScriptFrame* frame )
{
  SySoundDevice* pSoundDev = frame->mOwner->GetTitan()->GetScene()->GetSoundDev();
  if (pSoundDev)
  {
    pSoundDev->PlayStream( this->mSoundName.AsChar(), 127, SySoundDevice::kSoundClassFx );
  }
  return( SR_NEXT );
}

///////////////////////////////////////////////////////////////////////////// cScriptResponseGoTo

cScriptResponseGoTo::cScriptResponseGoTo()
{
  InitPropObject( SCRIPTRESPONSE_GOTO_CLASSID );

  mTarget = ST_SELF;
  mTargetLocation = 0;
}

int                  
cScriptResponseGoTo::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_GOTO_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseGoTo", 
               Creator, 
               ClassID, 
               0 ); 

  SyPropEnum *propEnum;
  AddEnumProperty(ClassID,0x1001,SyMemberOffset(cScriptResponseGoTo,mTarget),"mTarget",&propEnum);
  propEnum->Add(ST_SELF,"SELF");
  propEnum->Add(ST_ACTOR,"ACTOR");
  propEnum->Add(ST_SPECIFIED,"SPECIFIED");

  AddInt32Property(ClassID,0x1002,SyMemberOffset(cScriptResponseGoTo,mTargetLocation),"mTargetLocation");
  return 0;

}

SyPropObject*        
cScriptResponseGoTo::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseGoTo();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}
///////////////////////////////////////////////////////////////////////////// cScriptResponseSetAggro

cScriptResponseSetAggro::cScriptResponseSetAggro()
{
  InitPropObject( SCRIPTRESPONSE_SETAGGRO_CLASSID );

  mTarget = ST_SELF;
  mTargetLocation = 0;
  mAgro = 0;
}

int                  
cScriptResponseSetAggro::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_SETAGGRO_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseSetAggro", 
               Creator, 
               ClassID, 
               0 ); 

  SyPropEnum *propEnum;
  AddEnumProperty(ClassID,0x1001,SyMemberOffset(cScriptResponseSetAggro,mTarget),"mTarget",&propEnum);
  propEnum->Add(ST_SELF,"SELF");
  propEnum->Add(ST_ACTOR,"ACTOR");
  propEnum->Add(ST_SPECIFIED,"SPECIFIED");

  AddInt32Property(ClassID,0x1002,SyMemberOffset(cScriptResponseSetAggro,mTargetLocation),"mTargetLocation");
  AddInt32Property(ClassID,0x1003,SyMemberOffset(cScriptResponseSetAggro,mAgro),"mAgro");
  return 0;

}

SyPropObject*        
cScriptResponseSetAggro::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseSetAggro();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

//------------------------------------------------------------------------------------------Global


///////////////////////////////////////////////////////////////////////////// cScriptResponseTeleportTo

cScriptResponseTeleportTo::cScriptResponseTeleportTo()
{
  InitPropObject( SCRIPTRESPONSE_TELEPORTTO_CLASSID );

  mTargetLocation = 0;
}

int                  
cScriptResponseTeleportTo::InitPropClass()
{
  
  SyPropClassID ClassID = SCRIPTRESPONSE_TELEPORTTO_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseTeleportTo", 
               Creator, 
               ClassID, 
               0 ); 
  AddInt32Property(ClassID,0x1001,SyMemberOffset(cScriptResponseTeleportTo,mTargetLocation),"mTargetLocation");
  return 0;

}

SyPropObject*        
cScriptResponseTeleportTo::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseTeleportTo();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

///////////////////////////////////////////////////////////////////////////// cScriptResponseSpawn

cScriptResponseSpawn::cScriptResponseSpawn()
{
  InitPropObject( SCRIPTRESPONSE_SPAWN_CLASSID );

  mMaster = 0;
  mQuantity = 1;
  mTargetLocation = 0;
}

int                  
cScriptResponseSpawn::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_SPAWN_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseSpawn", 
               Creator, 
               ClassID, 
               0 ); 
  AddInt32Property(ClassID,0x1001,SyMemberOffset(cScriptResponseSpawn,mMaster),"mMaster");
  AddInt32Property(ClassID,0x1002,SyMemberOffset(cScriptResponseSpawn,mQuantity),"mQuantity");
  AddInt32Property(ClassID,0x1003,SyMemberOffset(cScriptResponseSpawn,mTargetLocation),"mTargetLocation");
  return 0;

}

SyPropObject*        
cScriptResponseSpawn::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseSpawn();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

///////////////////////////////////////////////////////////////////////////// cScriptResponseGiveItems

cScriptResponseGiveItem::cScriptResponseGiveItem()
{
  InitPropObject( SCRIPTRESPONSE_GIVEITEM_CLASSID );

  mTarget = ST_SELF;
  mItemID = 0;
  mQuantity = 1;
}

int                  
cScriptResponseGiveItem::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_GIVEITEM_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseGiveItem", 
               Creator, 
               ClassID, 
               0 ); 
  SyPropEnum *propEnum;
  AddEnumProperty(ClassID,0x1001,SyMemberOffset(cScriptResponseGiveItem,mTarget),"mTarget",&propEnum);
  propEnum->Add(ST_SELF,"SELF");
  propEnum->Add(ST_ACTOR,"ACTOR");
  propEnum->Add(ST_SPECIFIED,"SPECIFIED");

  AddInt32Property(ClassID,0x1002,SyMemberOffset(cScriptResponseGiveItem,mSpecified),"mSpecified");
  AddInt32Property(ClassID,0x1003,SyMemberOffset(cScriptResponseGiveItem,mItemID),"mItemID");
  AddInt32Property(ClassID,0x1004,SyMemberOffset(cScriptResponseGiveItem,mQuantity),"mQuantity");
  return 0;

}

SyPropObject*        
cScriptResponseGiveItem::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseGiveItem();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

///////////////////////////////////////////////////////////////////////////// cScriptResponseDelay

cScriptResponseDelay::cScriptResponseDelay()
{
  InitPropObject( SCRIPTRESPONSE_DELAY_CLASSID );

  mTime = 0.0f;
}

int                  
cScriptResponseDelay::InitPropClass()
{
  SyPropClassID ClassID = SCRIPTRESPONSE_DELAY_CLASSID;

/* Add the class */

  AddSubClass( ClassID, 
               SCRIPTRESPONSE_CLASSID,
               ClassID,
               "cScriptResponseDelay", 
               Creator, 
               ClassID, 
               0 ); 
  AddFloat32Property(ClassID,0x1001,SyMemberOffset(cScriptResponseDelay,mTime),"mTime");
  return 0;

}

SyPropObject*        
cScriptResponseDelay::Creator()
{
  SyPropObject *pObject;

  pObject = SyNew cScriptResponseDelay();
  if(pObject == NULL)
  {
    SyAssert(0);
    return(NULL);
  }

  return(pObject);
}

//-------------------------------------------------------------------------------- Global Functions
void 
RegPropClasses_Script()
{
  cScript::InitPropClass();
  cScriptBlock::InitPropClass();
  cScriptEventHandler::InitPropClass();
  cScriptResponse::InitPropClass();
  cScriptResponseCheckState::InitPropClass();
  cScriptResponseEndif::InitPropClass();
  cScriptResponseElse::InitPropClass();
  cScriptResponseChangeState::InitPropClass();
  cScriptResponsePlayMusic::InitPropClass();
  cScriptResponsePlaySound::InitPropClass();
  cScriptResponseGoTo::InitPropClass();
  cScriptResponseTeleportTo::InitPropClass();
  cScriptResponseSetAggro::InitPropClass();
  cScriptResponseSpawn::InitPropClass();
  cScriptResponseGiveItem::InitPropClass();
  cScriptResponseDelay::InitPropClass();
}

// EOF
