/********************************************************************

Module: TiAbilityTree

Author: Jon Wiesman

Description:

Class that handles dependencies for the ability tree.

Copyright 2006 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#ifndef __TiAbilityTree_h__
#define __TiAbilityTree_h__

#include "TiPtSizeRect.h"

enum EAbilityTreeNodeFlags
{
  ATN_BLANK           = 0x0000,
  ATN_ABILITY         = 0x0001,
  ATN_VERTBAR         = 0x0002,
  ATN_HORZBAR         = 0x0004,
  ATN_FROM_TOP        = ATN_VERTBAR,
  ATN_FROM_LEFT       = 0x0010,
  ATN_FROM_RIGHT      = 0x0020,
  ATN_TO_LEFT         = 0x0040,
  ATN_TO_RIGHT        = 0x0080,
};

class TiAbilityTreeNode
{
  uint32                  m_eFlags;
public:
  TiAbilityTreeNode();

  TiPoint                 m_ptPos;
  uint32                  m_uAbilityHash;
  int32                   m_nMinLevel;

  void                    Clear() {m_eFlags = 0; m_uAbilityHash = 0; m_nMinLevel = 0;}
  void                    AddConnector(int nConnector) {m_eFlags |= nConnector;}
  void                    ToggleConnector(int nConnector) {m_eFlags ^= nConnector;}
  void                    ClearConnector(int nConnector) {m_eFlags &= (~nConnector);}
  void                    SetAbility() {m_eFlags = ATN_ABILITY;}
  bool                    IsAbility() const {return ((m_eFlags & ATN_ABILITY) != 0);}
  bool                    IsBlank() const {return (m_eFlags == ATN_BLANK);}
  bool                    HasConnector(int nConnector) const {return ((m_eFlags & nConnector) != 0);}

  // for serializing
  int                     GetMinLevel() const {return m_nMinLevel;}
  void                    SetMinLevel(int nLevel) {m_nMinLevel = nLevel;}
  int                     GetFlags() const {return m_eFlags;}
  void                    SetFlags(uint32 nFlags) {m_eFlags = nFlags;}
  void                    SetAbilityHash(uint32 uHash) {m_uAbilityHash = uHash;}
};

const int32 c_nMaxATCols = 7;
const int32 c_nMaxATRows = 15;


class TiAbilityTree
{
  TiAbilityTreeNode  m_arNodes[c_nMaxATRows * c_nMaxATCols];

public:
  TiAbilityTree();
  void                        Reset();

  TiAbilityTreeNode          *GetNode(TiPoint pt);
  const TiAbilityTreeNode    *GetNode(TiPoint pt) const;

  bool                        IsNodeDependentOn(const TiAbilityTreeNode *pNode1, const TiAbilityTreeNode *pNode2) const;
  bool                        IsNodeDependentOn(TiPoint pt1, TiPoint pt2) const;
  bool                        MakeConnection(TiPoint pt1, TiPoint pt2);
  void                        LoadFile(const char8 *pszFilename);

  static void FillTestTree(TiAbilityTree &tree);
  static bool UnitTest();
};

#endif


