/********************************************************************

  Module: TiCharacterData

  Author: Carlo Ordonez

  Description:

    Updates character window with data from the game

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#ifndef _TiCharacterData_h_
#define _TiCharacterData_h_

#include "SyString.h"
#include "SyResourceID.h"
#include <vector>

class TitanI;

enum EEquipSlot  // see eEquipSlot
{
  kSlotMelee,
  kSlotRanged,
  kSlotHead,
  kSlotChest,
  kSlotShoulders,
  kSlotLegs,
  kSlotFeet,
  kSlotBelt,
  kSlotNecklace,
  kSlotLRing,
  kSlotRRing,
  kSlotUnequipped,
};

enum EItemType   // see eItemType
{
  kTypeMelee,
  kTypeRanged,
  kTypeHead,
  kTypeChest,
  kTypeShoulders,
  kTypeLegs,
  kTypeFeet,
  kTypeBelt,
  kTypeNecklace,
  kTypeRing,
  kTypeEnhancement,
};

class TiCharacterItemInfo
{
public:
  TiCharacterItemInfo();

  int32             m_iHandle;      // index back into game's inventory
  SyString          m_strName;
  SyString          m_strDescription;
  SyString          m_strIcon;
  SyResourceID      m_idIcon;
  EItemType         m_eType;
  EEquipSlot        m_eSlot;
  int32             m_nDamageBonus;
  int32             m_nPhysicalDefense;
  int32             m_nMagicalDefense;
  int32             m_nAttackPower;
  int32             m_nSpellPower;
};



class TiCharacterData
{
  enum 
  {
    kEnhancementSlotCount = 4,
  };
public:
  TiCharacterData();
  ~TiCharacterData() {}

  int32           Init( TitanI* pTitan );
  void            Update();

  void            SetPlayerCount( int32 Count );

  const SyString& GetName( int32 iPlayer ) const;
  const SyString& GetClassString( int32 iPlayer ) const;
  const SyString& GetExperience( int32 iPlayer ) const;
  const SyString& GetNextLevel( int32 iPlayer ) const;
  const SyString& GetHealth( int32 iPlayer ) const;
  const SyString& GetMana( int32 iPlayer ) const;
  const SyString& GetClassAbilityTreeFile(int32 iPlayer) const;
  int32           GetLevel(int32 iPlayer) const;
  int32           GetMaxHealth( int32 iPlayer ) const;
  int32           GetMaxMana( int32 iPlayer ) const;
  int32           GetAttackPower( int32 iPlayer ) const;
  int32           GetMagicPower( int32 iPlayer ) const;
  int32           GetMeleeDefense( int32 iPlayer ) const;
  int32           GetSpellDefense( int32 iPlayer ) const;

  int32           GetBonusPoints(int32 iPlayer ) const;
  int32           GetHealthPointsPerBonusPoint() const {return m_nManaPointsPerBonusPoint;}
  int32           GetManaPointsPerBonusPoint() const {return m_nHealthPointsPerBonusPoint;}

  int32           GetItemCount( int32 iPlayer ) const;
  void            ItemActivate( int32 iPlayer, int32 ItemHandle );
  void            ItemDrop( int32 iPlayer, int32 ItemHandle );
  int32           GetSlotEnhancement(int32 iPlayer, int32 iSlot) const;

  int32           GetAbilityPoints( int32 iPlayer ) const;
  const SyString  GetMappedAbility( int32 iPlayer, int32 Button ) const;
  void            AbilityMap( int32 iPlayer, int32 AbilityHandle, int32 Button );
  void            AbilityBuy( int32 iPlayer, int32 AbilityHandle );

  TitanI          *GetTitanInterface() const {return m_pTitan;}

  int32               GetPlayerCount() const {return m_nPlayerCount;}
  int32               GetCurrentItemHandle(int32 iPlayer, EEquipSlot eSlot) const;
  const SyString      GetCurrentItemName( int32 iPlayer, EEquipSlot eSlot ) const;
  const SyString      GetCurrentItemDescr( int32 iPlayer, EEquipSlot eSlot ) const;
  SyResourceID        GetCurrentItemIcon( int32 iPlayer, EEquipSlot eSlot) const;

  const std::vector< TiCharacterItemInfo >& GetInventory( int32 iPlayer ) const;
  const std::vector< TiCharacterItemInfo >& GetEquippableItems( int32 iPlayer, EEquipSlot Slot ) const;
  const std::vector< int32 >&               GetEnhancements(int32 iPlayer) const;

  class AbilityInfo
  {
  public:
    AbilityInfo();

    int32         m_iHandle;      // index back into game's spellbook
    SyString      m_strName;
    SyString      m_strDescription;
    int32         m_nCost;
    bool          m_bActive;
    bool          m_bBuyable;
  };
  const std::vector< AbilityInfo >& GetCurrentAbilities( int32 iPlayer ) const;
  const std::vector< AbilityInfo >& GetAvailableAbilities( int32 iPlayer ) const;

private:

  // internal class
  class PlayerInfo
  {
  public:
    PlayerInfo();

    // equipable items, by item type
    std::vector< TiCharacterItemInfo >  m_arMelee;
    std::vector< TiCharacterItemInfo >  m_arRanged;
    std::vector< TiCharacterItemInfo >  m_arHead;
    std::vector< TiCharacterItemInfo >  m_arChest;
    std::vector< TiCharacterItemInfo >  m_arShoulders;
    std::vector< TiCharacterItemInfo >  m_arLegs;
    std::vector< TiCharacterItemInfo >  m_arFeet;
    std::vector< TiCharacterItemInfo >  m_arBelt;
    std::vector< TiCharacterItemInfo >  m_arNecklace;
    std::vector< TiCharacterItemInfo >  m_arRing;
    std::vector< int32 >                m_arEnhancements; // array of indices into m_arInventory (all should be like this?)

    // entire inventory
    std::vector< TiCharacterItemInfo > m_arInventory;
    uint32                  m_uGold;

    // stats
    SyString        m_strName;
    SyString        m_strClass;
    SyString        m_strAbilityTreeFile;
    SyString        m_strExperience;
    SyString        m_strNextLevel;
    SyString        m_strHealth;
    SyString        m_strMana;
    int32           m_nLevel;
    int32           m_nMaxHealth;
    int32           m_nMaxMana;
    int32           m_nAttackPower;
    int32           m_nMagicPower;
    int32           m_nMeleeDefense;
    int32           m_nSpellDefense;
    int32           m_nBonusPoints;
    int32           m_arSlotEnhancements[kEnhancementSlotCount];  //  index into m_arEnhancements

    // spells & abilities
    std::vector< AbilityInfo >  m_arCurrentAbilities;
    std::vector< AbilityInfo >  m_arAvailableAbilities;
    uint32                      m_nAbilityPoints;
  };

  static const int32 PLAYER_MAX = 4;

  TitanI*         m_pTitan;
  int32           m_nPlayerCount;
  int32           m_nHealthPointsPerBonusPoint;
  int32           m_nManaPointsPerBonusPoint;

  PlayerInfo      m_arPlayers[PLAYER_MAX];
};

#endif

