/******************************************************************

  Module: TiChildFrameDraw.cpp

  Author: Jon Wiesman

  Description:

    child frame drawing class

  Copyright 2006 Sony Online Entertainment.  All rights reserved.

*******************************************************************/

#include "TiChildFrameDraw.h"
#include "TiUI.h"

TiBoxDraw TiChildFrameDraw::sm_bdFrame;
TiBoxDraw TiChildFrameDraw::sm_bdScrollbarBG;
TiBoxDraw TiChildFrameDraw::sm_bdScrollbarDown;
TiBoxDraw TiChildFrameDraw::sm_bdScrollbarUp;
TiSprite TiChildFrameDraw::sm_sprBackground;
TiSprite TiChildFrameDraw::sm_sprDownArrow;
TiSprite TiChildFrameDraw::sm_sprUpArrow;
TiSprite TiChildFrameDraw::sm_sprThumb;

const TiSize  c_sizeBackgroundDiff(3, 23);
const TiSize  c_sizeAdditionalScrollbarDiff(0, 5);
const TiSize  c_sizeScrollbarTabDiff(10, 8);
const int32   c_nScrollbarTabHeight = 15;

TiChildFrameDraw::TiChildFrameDraw(bool bScrollbar, int32 nRange) : m_bScrollbar(bScrollbar), m_nRange(nRange)
{
  m_nPos = 0;

  static bool bInit = false;
  if(!bInit)
  {
    sm_bdFrame.Initialize("InnerFrame");
    sm_bdScrollbarBG.Initialize("ScrollBG");
    sm_bdScrollbarUp.Initialize("ScrollbarUp");
    sm_bdScrollbarDown.Initialize("ScrollbarDown");
    sm_sprBackground.SetInfo("menu_dialog_child_fill.tga");
    sm_sprDownArrow.SetInfo("menu_dialog_scrollbar_arrow_down.tga");
    sm_sprUpArrow.SetInfo("menu_dialog_scrollbar_arrow_up.tga");
    sm_sprThumb.SetInfo("menu_dialog_scrollbar_button.tga");
    bInit = true;
  }
}

TiRect TiChildFrameDraw::GetContentRect(TiRect rectBox) const
{
  if(rectBox.GetSize().cx < 60 || rectBox.GetSize().cy < 60)
    return TiRect(0, 0, 0, 0);

  TiRect rectContent = rectBox;
  rectContent.AdjustSize(-c_sizeBackgroundDiff);

  if(m_bScrollbar)
  {
    rectContent.AdjustSize(-c_sizeAdditionalScrollbarDiff);
    rectContent.right = rectBox.right - c_sizeScrollbarTabDiff.cx - sm_sprThumb.GetSize().cx;
  }

  return rectContent;
}

void TiChildFrameDraw::Draw(TiRect rectBox, float fOpacity) const
{
  if(rectBox.GetSize().cx < 60 || rectBox.GetSize().cy < 60)
    return;

  TiRect rectBackground = rectBox;
  TiRect rectScrollUp = rectBox;
  TiRect rectScrollDown = rectBox;

  rectBackground.AdjustSize(-c_sizeBackgroundDiff);
  rectScrollUp.AdjustSize(-c_sizeScrollbarTabDiff);
  rectScrollDown.AdjustSize(-c_sizeScrollbarTabDiff);

  rectScrollUp.bottom = rectScrollUp.top + c_nScrollbarTabHeight;
  rectScrollDown.top = rectScrollDown.bottom - c_nScrollbarTabHeight;

  sm_bdFrame.Draw(rectBox, c_rectPS3Screen, fOpacity);

  sm_sprBackground.Stretch(rectBackground, c_rectPS3Screen, fOpacity);

  if(m_bScrollbar)
  {
    sm_bdScrollbarUp.Draw(rectScrollUp);
    sm_bdScrollbarDown.Draw(rectScrollDown);

    TiPoint ptArrow((rectScrollUp.Width() - sm_sprUpArrow.GetSize().cx) / 2, (rectScrollUp.Height() - sm_sprUpArrow.GetSize().cy) / 2);
    ptArrow += rectScrollUp.TopLeft(); 
    sm_sprUpArrow.Draw(ptArrow, c_rectPS3Screen, (m_nPos > 0) ? fOpacity : 0.5f * fOpacity);

    ptArrow.Set((rectScrollDown.Width() - sm_sprDownArrow.GetSize().cx) / 2, (rectScrollDown.Height() - sm_sprDownArrow.GetSize().cy) / 2);
    ptArrow += rectScrollDown.TopLeft(); 
    sm_sprDownArrow.Draw(ptArrow, c_rectPS3Screen, (m_nPos < m_nRange) ? fOpacity : 0.5f * fOpacity);
    
    TiRect rectScrollbar = rectBackground;
    rectScrollbar.AdjustSize(-c_sizeAdditionalScrollbarDiff);
    rectScrollbar.right = rectScrollDown.right;
    rectScrollbar.left = rectScrollbar.right - sm_sprThumb.GetSize().cx;
    TiUI::Instance()->FillRectangle(rectScrollbar, SyColor32F(0, 0, 0, fOpacity));

    if(m_nRange > 0)
    {
      int32 nScrollHeight = rectScrollbar.Height() - sm_sprThumb.GetSize().cy;
      int32 nOffset = m_nPos * nScrollHeight / m_nRange;

      TiPoint ptThumb = rectScrollbar.TopLeft();
      ptThumb.y += nOffset;
      sm_sprThumb.Draw(ptThumb, c_rectPS3Screen, fOpacity);
    }
  }
}

