/********************************************************************

  Module: TiEnhancementsWindow

  Author: Jon Wiesman

  Description:

    Enhancements window 

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiEnhancementsWindow.h"

#include "InputConsole.h"
#include "TiCharacterData.h"
#include "TiFont.h"
#include "TiMessageBox.h"
#include "TiUI.h"
#include "TiWindowMessage.h"

static TiRect c_rectOuterFrame(TiPoint(12, 4), TiSize(457, 618));
static TiRect c_rectTitlebar(TiPoint(28, 11), TiSize(425, 27));


static TiRect c_rectActiveFrame(TiPoint(28, 200), TiSize(425, 60));
static TiRect c_rectFirstSlot(TiPoint(69, 206), TiSize(48, 48));
static int32 c_nSlotSpacing = 99;
static TiRect c_rectGridFrame(TiPoint(28, 270), TiSize(425, 326));
static TiRect c_rectFirstGrid(TiPoint(48, 285), TiSize(48, 48));
static TiSize c_sizeGridSpacing(68, 62);
const TiRect c_rectEnhancementName(TiPoint(102, 42), TiSize(357, 27));
const TiRect c_rectEnhancementDescr(TiPoint(102, 69), TiSize(357, 180));


//***********************************************************************
// Constructor/Destructor
//***********************************************************************

TiEnhancementsWindow::TiEnhancementsWindow( TiWindow* pParent, TiCharacterData* pData, int32 PlayerId )
: TiCharacterPage( pParent, pData, PlayerId, kEnhancements)
{
	m_eMode = EWMode_BrowsingSlots;
	m_cfdGrid.SetScrollbar(false);
	m_bdActive.Initialize("InnerFrame");
  m_sprEnhancementSlots.SetInfo("ui_enhancement_slots.tga");
}

TiEnhancementsWindow::~TiEnhancementsWindow()
{
}

//***********************************************************************
// Init
//***********************************************************************

int32 TiEnhancementsWindow::Init()
{
	if(( m_eMode != EWMode_ConfirmingSlot ) && ( m_eMode != EWMode_ConfirmingGrid ))
	{
		m_ptCurrent.Set(0, 0);
		m_iSlot = 0;
		m_eMode = EWMode_BrowsingSlots;
		m_iEnhancementToRemove = -1;
	}
	return( 0 );
}

/**********************************************************
* ProcessMessage
**********************************************************/
void TiEnhancementsWindow::ProcessMessage( TiWindowMessage* pMessage )
{
	switch( pMessage->GetMessageId() )
	{
		case TIWINDOW_SETFOCUS:
		{
			if( pMessage->GetSender() == this )
			{
				// TiCharacterWindow should have focus, not me
				GetParent()->ProcessMessage( pMessage );
				return;
			}
		}
	}
	TiWindow::ProcessMessage( pMessage );
}

//***********************************************************************
// DrawSelf
//***********************************************************************

void TiEnhancementsWindow::DrawSelf()
{
  DrawBackground();
  TiRect rectWindow = GetScreenRect();

  TiFont font(TiUI::Instance()->GetFontHandle("Default"));

  sm_bdFrame.DrawWithFill(c_rectOuterFrame + rectWindow.TopLeft(), c_crFrameFill);

  TiRect rect = c_rectTitlebar + rectWindow.TopLeft();
  char8 szBuffer[256] = "";
  static SyResourceID idEquipmentTabName = SyHashResourceID("UiEnhancementsTabName");

  m_pData->GetTitanInterface()->T4Expand(szBuffer, sizeof(szBuffer), idEquipmentTabName);
  font.Draw(szBuffer, c_rectTitlebar + GetScreenRect().TopLeft(), c_crWhite, TI_TF_SHADOW | TI_TF_HALIGN_CENTER);

  m_bdActive.DrawWithFill(c_rectActiveFrame + rectWindow.TopLeft(), c_crBlack);
  m_cfdGrid.Draw(c_rectGridFrame + rectWindow.TopLeft());

  m_sprEnhancementSlots.Draw(c_rectActiveFrame.TopLeft() + rectWindow.TopLeft());

  int32 iEnhancementToDisplay = -1;

  TiRect rectSlot = c_rectFirstSlot + rectWindow.TopLeft();
  for(int i = 0; i < EW_EnhancementSlots; i++)
  {
    if(i == m_iSlot && m_eMode != EWMode_BrowsingGrid)
    {
      TiRect rectSelect = rectSlot;
      rectSelect.AdjustSize(8, 8);
      sm_bdSelection.Draw(rectSelect);
      if(m_eMode == EWMode_BrowsingSlots)
      {
        iEnhancementToDisplay = m_pData->GetSlotEnhancement(m_iPlayer, i);
      }
    }

    TiUI::Instance()->FillRectangle(rectSlot, c_crWhite);
    TiRect rectInner = rectSlot;
    rectInner.AdjustSize(-1, -1);

    if(m_pData->GetSlotEnhancement(m_iPlayer, i) >= 0)
    {
      TiUI::Instance()->FillRectangle(rectInner, SyColor32F(.5f, .5f, .9f, 1.f));
    }
    else
    {
      TiUI::Instance()->FillRectangle(rectInner, c_crBlack);
    }

    rectSlot += TiPoint(c_nSlotSpacing, 0);
  }

  TiRect rectGrid = c_rectFirstGrid + rectWindow.TopLeft();
  TiPoint pt;
  const std::vector< int32 > &arEnhancements = m_pData->GetEnhancements(m_iPlayer);
  for(pt.y = 0; pt.y < EW_EnhancementGridHeight; pt.y++)
  {
    for(pt.x = 0; pt.x < EW_EnhancementGridWidth; pt.x++)
    {
      int32 iEnhancement = pt.y * EW_EnhancementGridWidth + pt.x;

      if(m_ptCurrent == pt && m_eMode != EWMode_BrowsingSlots)
      {
        TiRect rectSelect = rectGrid;
        rectSelect.AdjustSize(8, 8);
        sm_bdSelection.Draw(rectSelect);

        if(m_eMode == EWMode_BrowsingGrid && iEnhancement < (int32)(arEnhancements.size()))
        {
          iEnhancementToDisplay = arEnhancements[iEnhancement];
        }
      }

      if(iEnhancement < (int32)arEnhancements.size())
        TiUI::Instance()->FillRectangle(rectGrid, SyColor32F(.5f, .5f, .9f, 1.f));
      else
        TiUI::Instance()->FillRectangle(rectGrid, c_crBlack);
      rectGrid += TiPoint(c_sizeGridSpacing.cx, 0);
    }
    rectGrid.left = c_rectFirstGrid.left + rectWindow.left;
    rectGrid.right = rectGrid.left + c_rectFirstGrid.Width();
    rectGrid += TiPoint(0, c_sizeGridSpacing.cy);
  }

  if(iEnhancementToDisplay >= 0)
  {
    const TiCharacterItemInfo &item = m_pData->GetInventory(m_iPlayer)[iEnhancementToDisplay];

    TiRect rectText = c_rectEnhancementName + rectWindow.TopLeft();
    font.Draw(item.m_strName, rectText, c_crWhite, TI_TF_HALIGN_CENTER | TI_TF_SHADOW);
  }
}

//***********************************************************************
// ProcessKeyDown
//***********************************************************************

bool TiEnhancementsWindow::ProcessKeyDown( int32 KeyCode )
{
  switch( KeyCode )
  {
  case SyInputEvent::KEYCODE_UP:
    return( true );

  case SyInputEvent::KEYCODE_DOWN:
    return( true );
  }
  return( false );
}

//***********************************************************************
// ProcessButtonDown
//***********************************************************************

bool TiEnhancementsWindow::ProcessButtonDown( int32 DeviceId, int32 ButtonId )
{
  if( DeviceId != m_iPlayer ) { return( false ); }

  switch( TiWindow::GetButton( ButtonId ) )
  {
  case TiButtonMap::kUp:
    if(m_eMode == EWMode_BrowsingSlots || m_eMode == EWMode_SelectingSlot)
    {
      // ignore
    }
    else if(m_eMode == EWMode_BrowsingGrid || m_eMode == EWMode_SelectingGrid)
    {
      OnGridUp();
    }
    return( true );

  case TiButtonMap::kDown:
    if(m_eMode == EWMode_BrowsingSlots || m_eMode == EWMode_SelectingSlot)
    {
      OnSlotDown();
    }
    else if(m_eMode == EWMode_BrowsingGrid || m_eMode == EWMode_SelectingGrid)
    {
      OnGridDown();
    }
    return( true );

  case TiButtonMap::kLeft:
    if(m_eMode == EWMode_BrowsingSlots || m_eMode == EWMode_SelectingSlot)
    {
      OnSlotLeft();
    }
    else if(m_eMode == EWMode_BrowsingGrid || m_eMode == EWMode_SelectingGrid)
    {
      OnGridLeft();
    }
    return( true );

  case TiButtonMap::kRight:
    if(m_eMode == EWMode_BrowsingSlots || m_eMode == EWMode_SelectingSlot)
    {
      OnSlotRight();
    }
    else if(m_eMode == EWMode_BrowsingGrid || m_eMode == EWMode_SelectingGrid)
    {
      OnGridRight();
    }
    return( true );
    
  case TiButtonMap::kX:
    if(m_eMode == EWMode_BrowsingSlots)
    {
      m_eMode = EWMode_SelectingGrid;
    }
    else if(m_eMode == EWMode_BrowsingGrid)
    {
      m_eMode = EWMode_SelectingSlot;
	  m_iSlot = 0;
    }
    else if(m_eMode == EWMode_SelectingGrid)
    {
      OnSelectGrid();
    }
    else if(m_eMode == EWMode_SelectingSlot)
    {
      OnSelectSlot();
    }
    return true;

  case TiButtonMap::kCircle:
	  if(m_eMode == EWMode_SelectingGrid)
	  {
		  m_eMode = EWMode_BrowsingSlots;
		  return true;
	  }
	  else if(m_eMode == EWMode_SelectingSlot)
	  {
		  m_eMode = EWMode_BrowsingGrid;
		  return true;
	  }

  default:
    return( false );
  }
}

bool TiEnhancementsWindow::UseEnhancement( bool bConfirmed )
{
	// validate currently selected enhancement
	int iEnhancement = m_ptCurrent.y * EW_EnhancementGridWidth + m_ptCurrent.x;
	const std::vector< int32 > &arEnhancements = m_pData->GetEnhancements(m_iPlayer); 
	if(iEnhancement >= (int32)arEnhancements.size())
	{
		return false;
	}

	// make sure it can go in current slot
	int iInventoryItem = arEnhancements[iEnhancement];
	if( !m_pData->GetTitanInterface()->InventoryGetIsEnhancementValidInSlot( m_iPlayer, iInventoryItem, m_iSlot ) )
	{
		return false;
	}


	// if not yet confirmed, store off previous enhancement
	if( !bConfirmed )
	{
		m_iEnhancementToRemove = m_pData->GetSlotEnhancement( m_iPlayer, m_iSlot );
	}


	// equip it
	m_pData->GetTitanInterface()->InventoryUseEnhancement(m_iPlayer, iInventoryItem, m_iSlot, bConfirmed);
	m_pData->Update();

	return true;
}

void TiEnhancementsWindow::OnSelectGrid()
{
	if( UseEnhancement( false ) )
	{
		m_eMode = EWMode_ConfirmingGrid;
		TiMessageBox mbox( this, "Put this enhancement in this slot?", "Confirmation", TiMessageBox::kYesNo );
		mbox.SetWidth( c_rectGridFrame.Width() );
		SyVect2I vCenter;
		vCenter.X = c_rectGridFrame.left + c_rectGridFrame.Width()/2 + GetScreenRect().left;
		vCenter.Y = c_rectGridFrame.top + c_rectGridFrame.Height()/2 + GetScreenRect().top;
		mbox.SetCenterPoint( vCenter );
		int nRetCode = mbox.ShowModal();
		if( nRetCode == TiMessageBox::kIdYes )
		{
			UseEnhancement( true );
			m_eMode = EWMode_BrowsingSlots;
		}
		else
		{
			m_eMode = EWMode_SelectingGrid;
			// reset old enhancement
			if( m_iEnhancementToRemove >= 0 )
			{
				m_pData->GetTitanInterface()->InventoryUseEnhancement(m_iPlayer, m_iEnhancementToRemove, m_iSlot, false);
			}
			else
			{
				m_pData->GetTitanInterface()->InventoryClearEnhancement( m_iPlayer, m_iSlot );
			}
			m_pData->Update();
		}
		m_iEnhancementToRemove = -1;
	}
}

void TiEnhancementsWindow::OnSelectSlot()
{
	if( UseEnhancement( false ) )
	{
		m_eMode = EWMode_ConfirmingSlot;
		TiMessageBox mbox( this, "Put this enhancement in this slot?", "Confirmation", TiMessageBox::kYesNo );
		mbox.SetWidth( c_rectGridFrame.Width() );
		SyVect2I vCenter;
		vCenter.X = c_rectGridFrame.left + c_rectGridFrame.Width()/2 + GetScreenRect().left;
		vCenter.Y = c_rectGridFrame.top + c_rectGridFrame.Height()/2 + GetScreenRect().top;
		mbox.SetCenterPoint( vCenter );
		int nRetCode = mbox.ShowModal();
		if( nRetCode == TiMessageBox::kIdYes )
		{
			UseEnhancement( true );
			m_eMode = EWMode_BrowsingGrid;
		}
		else
		{
			m_eMode = EWMode_SelectingSlot;
			// reset old enhancement
			if( m_iEnhancementToRemove >= 0 )
			{
				m_pData->GetTitanInterface()->InventoryUseEnhancement(m_iPlayer, m_iEnhancementToRemove, m_iSlot, false);
			}
			else
			{
				m_pData->GetTitanInterface()->InventoryClearEnhancement( m_iPlayer, m_iSlot );
			}
			m_pData->Update();
		}
		m_iEnhancementToRemove = -1;
	}
}

void TiEnhancementsWindow::OnGridDown()
{
  m_ptCurrent.y = SY_MIN(m_ptCurrent.y + 1, EW_EnhancementGridHeight - 1);
}

void TiEnhancementsWindow::OnGridUp()
{
  if(m_ptCurrent.y == 0)
  {
    if(m_eMode == EWMode_BrowsingGrid)
      m_eMode = EWMode_BrowsingSlots;
  }
  else
  {
    m_ptCurrent.y--;
  }
}

void TiEnhancementsWindow::OnGridLeft()
{
	m_ptCurrent.x = SY_MAX(0, m_ptCurrent.x - 1);
}

void TiEnhancementsWindow::OnGridRight()
{
	m_ptCurrent.x = SY_MIN(m_ptCurrent.x + 1, EW_EnhancementGridWidth - 1);
}

void TiEnhancementsWindow::OnSlotDown()
{
	if(m_eMode == EWMode_BrowsingSlots)
	{
		m_eMode = EWMode_BrowsingGrid;
	}
}

void TiEnhancementsWindow::OnSlotLeft()
{
	// if selecting a slot, only 2 will be valid, and one of them is always #0
	if( m_eMode == EWMode_SelectingSlot )
	{
		m_iSlot = 0;
	}
	else
	{
		// just move to previous slot
		m_iSlot = SY_MAX(0, m_iSlot - 1);
	}
}

void TiEnhancementsWindow::OnSlotRight()
{
	if( m_eMode == EWMode_SelectingSlot )
	{
		// if selecting a slot, only 2 will be valid, select the one that is NOT #0

		// if we're not in slot 0, we're already as far right as we can go
		if( m_iSlot != 0 )
		{
			return;
		}

		// get inventory item index from current location
		int iEnhancement = m_ptCurrent.y * EW_EnhancementGridWidth + m_ptCurrent.x;
		const std::vector< int32 > &arEnhancements = m_pData->GetEnhancements(m_iPlayer); 
		SyAssert(iEnhancement < (int32)arEnhancements.size());
		int iInventoryItem = arEnhancements[iEnhancement];

		// find the other slot this enhancement can go in
		for( int iSlot=1; iSlot<EW_EnhancementSlots; ++iSlot )
		{
			if( m_pData->GetTitanInterface()->InventoryGetIsEnhancementValidInSlot( m_iPlayer, iInventoryItem, iSlot ) )
			{
				m_iSlot = iSlot;
				return;
			}
		}
		// should never happen
		SyAssert( false );
	}
	else
	{
		// just move to next slot
		m_iSlot = SY_MIN(m_iSlot + 1, EW_EnhancementSlots - 1);
	}
}

