/********************************************************************

  Module: TiMessageBox

  Author: Carlo Ordonez

  Description:

    Message Box Dialog

  Copyright 2006 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiMessageBox.h"

#include "SyResourceID.h"
#include "TiFocusMessage.h"
#include "TitanUI.h"
#include "TiUIControl.h"

//***********************************************************************
// Constructors/Destructor
//***********************************************************************

TiMessageBox::TiMessageBox( TiApp* pApp, const char8* pText, const char8* pCaption, Type type )
	: TiDialog( pApp )
	, mText( pText )
	, mCaption( pCaption )
	, mType( type )
	, m_vCenterPoint( -1, -1 )
	, m_nWidth( 500 )
{
	mpCtrl = SyNew TiUIControl();
}

TiMessageBox::TiMessageBox( TiWindow* pParent, const char8* pText, const char8* pCaption, Type type )
	: TiDialog( pParent )
	, mText( pText )
	, mCaption( pCaption )
	, mType( type )
	, m_vCenterPoint( -1, -1 )
	, m_nWidth( 500 )
{
	mpCtrl = SyNew TiUIControl();
}

TiMessageBox::~TiMessageBox()
{
  delete mpCtrl;
}

//***********************************************************************
// DrawSelf
//***********************************************************************

void TiMessageBox::DrawSelf()
{
  int32 margin = 20;
  int32 tbWidth = m_nWidth - (margin<<1);
  int32 rows = mpCtrl->CountLines( mText.AsChar(), tbWidth ) + 5;

  // frame
  if( m_vCenterPoint != SyVect2I(-1,-1) )
  {
	  mpCtrl->Init( m_vCenterPoint, m_nWidth, rows );
  }
  else
  {
	  mpCtrl->Init( m_nWidth, rows );
  }
  mpCtrl->Draw();

  // caption
  mpCtrl->DrawTextBox( mCaption.AsChar(), margin, 0, tbWidth, 1, TiUIControl::kNormal, TI_TF_HALIGN_CENTER|TI_TF_SHADOW );

  // text
  mpCtrl->DrawTextBox( mText.AsChar(), margin, 2, tbWidth, rows - 4, TiUIControl::kNormal, TI_TF_HALIGN_CENTER|TI_TF_SHADOW );


  static SyResourceID idAbortRetryCancel	= SyHashResourceID("UiPromptAbortRetryCancel");
  static SyResourceID idOk					= SyHashResourceID("UiPromptOk");
  static SyResourceID idOkCancel			= SyHashResourceID("UiPromptOkCancel");
  static SyResourceID idRetryCancel			= SyHashResourceID("UiPromptRetryCancel");
  static SyResourceID idYesNo				= SyHashResourceID("UiPromptYesNo");
  static SyResourceID idYesNoCancel			= SyHashResourceID("UiPromptYesNoCancel");

  char8 szBuffer[256] = "";

  // buttons
  switch( mType )
  {
  case kAbortRetryIgnore:
	  TitanUI::Instance()->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), idAbortRetryCancel);
	break;
  case kOk:
	  TitanUI::Instance()->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), idOk);
    break;
  case kOkCancel:
	  TitanUI::Instance()->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), idOkCancel);
    break;
  case kRetryCancel:
	  TitanUI::Instance()->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), idRetryCancel);
    break;
  case kYesNo:
	  TitanUI::Instance()->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), idYesNo);
    break;
  case kYesNoCancel:
	  TitanUI::Instance()->GetTitanI()->T4Expand(szBuffer, sizeof(szBuffer), idYesNoCancel);
    break;
  default:
    SyAssertf( 0, "Unhandled dialog type" );
    break;
  }
  mpCtrl->DrawTextBox( szBuffer, margin, rows-2, tbWidth, 1, TiUIControl::kNormal, TI_TF_HALIGN_CENTER|TI_TF_SHADOW );

}

//***********************************************************************
// ProcessButtonDown
//***********************************************************************

bool TiMessageBox::ProcessButtonDown( int32 DeviceId, int32 ButtonId )
{
  TiButtonMap::Button button = TiWindow::GetButton( ButtonId );

  if( mType == kOk )
  {
    switch( button )
    {
    case TiButtonMap::kX:
      EndModal( kIdOk );
      return( true );
    default:
      return( false );
    }
  }
  else if( mType == kAbortRetryIgnore )
  {
    switch( button )
    {
    case TiButtonMap::kSquare:
      EndModal( kIdAbort );
      return( true );
    case TiButtonMap::kX:
      EndModal( kIdRetry );
      return( true );
    case TiButtonMap::kCircle:
      EndModal( kIdIgnore );
      return( true );
    default:
      return( false );
    }
  }
  else if( mType == kOkCancel )
  {
    switch( button )
    {
    case TiButtonMap::kX:
      EndModal( kIdOk );
      return( true );
    case TiButtonMap::kCircle:
      EndModal( kIdCancel );
      return( true );
    default:
      return( false );
    }
  }
  else if( mType == kRetryCancel )
  {
    switch( button )
    {
    case TiButtonMap::kX:
      EndModal( kIdRetry );
      return( true );
    case TiButtonMap::kCircle:
      EndModal( kIdCancel );
      return( true );
    default:
      return( false );
    }
  }
  else if( mType == kYesNo )
  {
    switch( button )
    {
    case TiButtonMap::kX:
      EndModal( kIdYes );
      return( true );
    case TiButtonMap::kCircle:
      EndModal( kIdNo );
      return( true );
    default:
      return( false );
    }
  }
  else if( mType == kYesNoCancel )
  {
    switch( button )
    {
    case TiButtonMap::kX:
      EndModal( kIdYes );
      return( true );
    case TiButtonMap::kCircle:
      EndModal( kIdNo );
      return( true );
    case TiButtonMap::kTriangle:
      EndModal( kIdCancel );
      return( true );
    default:
      return( false );
    }
  }
  else
  {
    SyAssertf( 0, "Unhandled dialog type" );
    return( false );
  }
}

//***********************************************************************
// ProcessMessage
//***********************************************************************

void TiMessageBox::ProcessMessage( TiWindowMessage* pMessage )
{
  switch( pMessage->GetMessageId() )
  {
  case TIWINDOW_SETFOCUS:
    {
      TiFocusMessage* pMsg = static_cast<TiFocusMessage*>(pMessage);
      TiWindow* pNew = pMsg->GetNewWindow();

      if( pNew == this )
      {
        mpCtrl->Close( true );
        mpCtrl->Open();
      }
    }
    break;
  }
}
