/********************************************************************

  Module: TiStatsWindow

  Author: Carlo Ordonez

  Description:

    Character Stats Window

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#include "TiStatsWindow.h"
#include "TiCharacterData.h"
#include "TiUI.h"
#include "TiFont.h"
#include "t4utf.h"

static TiRect c_rectOuterFrame(TiPoint(12, 4), TiSize(457, 618));
static TiRect c_rectInnerFrame(TiPoint(28, 200), TiSize(425, 396));
static TiRect c_rectFirstStat(TiPoint(32, 212), TiSize(417, 62));
static TiRect c_rectFirstStatText(TiPoint(104, 212), TiSize(200, 62));
static TiRect c_rectFirstStatBase(TiPoint(304, 212), TiSize(64, 62));
static TiRect c_rectFirstStatAdjustment(TiPoint(368, 212), TiSize(64, 62));
static TiPoint c_ptLeftArrowOffset1(326, 21);
static TiPoint c_ptLeftArrowOffset2(330, 21);
static TiPoint c_ptRightArrowOffset1(398, 21);
static TiPoint c_ptRightArrowOffset2(394, 21);
static TiRect c_rectAdjustmentColumn(TiPoint(368, 212), TiSize(64, 372));
static TiRect c_rectTitlebar(TiPoint(28, 11), TiSize(425, 27));
static TiPoint c_ptIconOffset(16, 15);

//***********************************************************************
// Constructor/Destructor
//***********************************************************************

TiStatsWindow::TiStatsWindow( TiWindow* pParent, TiCharacterData* pData, int32 PlayerId )
: TiCharacterPage( pParent, pData, PlayerId, kStats )
{
  m_cfd.SetScrollbar(false);

  m_sprStatIcons[SW_Health].SetInfo("stats_health.tga");
  m_sprStatIcons[SW_Mana].SetInfo("stats_icon_mana.tga");
  m_sprStatIcons[SW_AttackPower].SetInfo("stats_icon_attackpower.tga");
  m_sprStatIcons[SW_MagicPower].SetInfo("stats_icon_magicpower.tga");
  m_sprStatIcons[SW_MeleeDefense].SetInfo("stats_icon_meleedefense.tga");
  m_sprStatIcons[SW_SpellDefense].SetInfo("stats_icon_spelldefense.tga");
  m_sprArrowLeft.SetInfo("arrow_left.tga");
  m_sprArrowRight.SetInfo("arrow_right.tga");

  for(int i = 0; i < SW_MaxStats; i++)
  {
    m_nStats[i] = 0;
    m_nAdjustments[i] = 0;
    m_nDeltaPerPoint[i] = 1;
  }
  m_nBonusPoints = 0;
  m_bLevelingUp = false;
  m_iStat = SW_Health;
}

TiStatsWindow::~TiStatsWindow()
{
}


int32 TiStatsWindow::Init()
{
  TiCharacterPage::Init();

  m_nDeltaPerPoint[SW_Health] = m_pData->GetHealthPointsPerBonusPoint();
  m_nDeltaPerPoint[SW_Mana] = m_pData->GetManaPointsPerBonusPoint();

  m_iStat = SW_Health;

  if(m_iPlayer >= 0 && m_iPlayer < m_pData->GetPlayerCount())
  {
    m_nStats[SW_Health] = m_pData->GetMaxHealth(m_iPlayer);
    m_nStats[SW_Mana] = m_pData->GetMaxMana(m_iPlayer);
    m_nStats[SW_AttackPower] = m_pData->GetAttackPower(m_iPlayer);
    m_nStats[SW_MagicPower] = m_pData->GetMagicPower(m_iPlayer);
    m_nStats[SW_MeleeDefense] = m_pData->GetMeleeDefense(m_iPlayer);
    m_nStats[SW_SpellDefense] = m_pData->GetSpellDefense(m_iPlayer);
    memset(m_nAdjustments, 0, sizeof(m_nAdjustments));
    m_nBonusPoints = m_pData->GetBonusPoints(m_iPlayer);
  }
  m_bLevelingUp = (m_nBonusPoints > 0);
  return 0;
}


//***********************************************************************
// DrawSelf
//***********************************************************************

void TiStatsWindow::DrawSelf()
{
  const char8 *pszStatNameIds[SW_MaxStats] = {
    "UiStatNameHealth", 
    "UiStatNameMana", 
    "UiStatNameAttackPower", 
    "UiStatNameMagicPower", 
    "UiStatNameMeleeDefense", 
    "UiStatNameSpellDefense",
  };
  const char8 *pszStatDescriptions[SW_MaxStats] = {
    "UiStatDescrHealth", 
    "UiStatDescrMana", 
    "UiStatDescrAttackPower", 
    "UiStatDescrMagicPower", 
    "UiStatDescrMeleeDefense", 
    "UiStatDescrSpellDefense",
  };

  TiFont font(TiUI::Instance()->GetFontHandle("Default"));
  char8 szString[300];

  DrawBackground();

  TiRect rectWindow = GetScreenRect();

  sm_bdFrame.DrawWithFill(c_rectOuterFrame + rectWindow.TopLeft(), c_crFrameFill);
  m_cfd.Draw(c_rectInnerFrame + rectWindow.TopLeft());
  

  TiRect rect = c_rectTitlebar + rectWindow.TopLeft();
  m_pData->GetTitanInterface()->T4Expand(szString, sizeof(szString), SyHashResourceID("UiStatsTabName"));
  font.Draw(szString, rect, c_crWhite, TI_TF_SHADOW | TI_TF_HALIGN_CENTER);

  rect = sm_bdFrame.GetInternalRect(c_rectOuterFrame + rectWindow.TopLeft());
  rect.bottom = rect.top + font.GetHeight();

  font.Draw(m_pData->GetName(m_iPlayer), rect, c_crWhite, TI_TF_SHADOW);
  font.Draw(m_pData->GetClassString(m_iPlayer), rect, c_crWhite, TI_TF_SHADOW | TI_TF_HALIGN_CENTER);
  m_pData->GetTitanInterface()->T4SetContext('0', 0, T4::UNKNOWN_SEX, m_pData->GetLevel(m_iPlayer));
  m_pData->GetTitanInterface()->T4Expand(szString, sizeof(szString), SyHashResourceID("UiLevelFormat"));
  font.Draw(szString, rect, c_crWhite, TI_TF_SHADOW | TI_TF_HALIGN_RIGHT);
  

  rect += TiPoint(0, font.GetHeight());
  if(m_bLevelingUp)
  {
    TiUI::Instance()->FillRectangle(c_rectAdjustmentColumn + rectWindow.TopLeft(), SyColor32F(0.f, 0.f, 0.f, .9f));

    m_pData->GetTitanInterface()->T4Expand(szString, sizeof(szString), SyHashResourceID("UiLevelUp"));
    font.Draw(szString, rect, c_crWhite, TI_TF_SHADOW);
    m_pData->GetTitanInterface()->T4SetContext('0', 0, T4::UNKNOWN_SEX, m_nBonusPoints);
    m_pData->GetTitanInterface()->T4Expand(szString, sizeof(szString), SyHashResourceID("UiAvailablePointsFormat"));
    font.Draw(szString, rect, c_crWhite, TI_TF_SHADOW | TI_TF_HALIGN_RIGHT);
  }
 
  rect += TiPoint(0, font.GetHeight());
  rect.bottom = c_rectInnerFrame.bottom + rectWindow.bottom;
  m_pData->GetTitanInterface()->T4Expand(szString, sizeof(szString), SyHashResourceID(pszStatNameIds[m_iStat]));
  rect.top += font.Draw(szString, rect, c_crWhite, TI_TF_SHADOW | TI_TF_HALIGN_CENTER);
  m_pData->GetTitanInterface()->T4Expand(szString, sizeof(szString), SyHashResourceID(pszStatDescriptions[m_iStat]));
  font.Draw(szString, rect, c_crWhite, TI_TF_SHADOW | TI_TF_HALIGN_CENTER);

  rect = c_rectFirstStat + rectWindow.TopLeft();
  TiRect rectName = c_rectFirstStatText + rectWindow.TopLeft();
  TiRect rectBase = c_rectFirstStatBase + rectWindow.TopLeft();
  TiRect rectAdj = c_rectFirstStatAdjustment + rectWindow.TopLeft();
  for(int iStat = 0; iStat < SW_MaxStats; iStat++)
  {
    if(iStat == m_iStat)
    {
      sm_bdSelection.Draw(rect);

      TiRect rectSprite(rect.TopLeft() + c_ptIconOffset, m_sprStatIcons[iStat].GetSize());
      m_sprStatIcons[iStat].Pulse(rectSprite, .2f, 3.f * SY_PI);

      if(m_bLevelingUp)
      {
        m_sprArrowLeft.Oscillate(rect.TopLeft() + c_ptLeftArrowOffset1, rect.TopLeft() + c_ptLeftArrowOffset2, 9.f);
        m_sprArrowRight.Oscillate(rect.TopLeft() + c_ptRightArrowOffset1, rect.TopLeft() + c_ptRightArrowOffset2, 9.f);
      }
    }
    else
    {
      m_sprStatIcons[iStat].Draw(rect.TopLeft() + c_ptIconOffset);
    }

    m_pData->GetTitanInterface()->T4Expand(szString, sizeof(szString), SyHashResourceID(pszStatNameIds[iStat]));
    font.Draw(szString, rectName, c_crWhite, TI_TF_SHADOW | TI_TF_VCENTER);
    
    char8 szNumber[10];

    SyColor32F crText = c_crWhite;

    if(m_nAdjustments[iStat] > 0)
      crText = c_crGreen;

    if(m_bLevelingUp)
    {
      sprintf(szNumber, "%d", m_nStats[iStat] + m_nAdjustments[iStat] * m_nDeltaPerPoint[iStat]);
      font.Draw(szNumber, rectBase, crText, TI_TF_HALIGN_CENTER | TI_TF_SHADOW | TI_TF_VCENTER);

      sprintf(szNumber, "%d", m_nAdjustments[iStat]);
      font.Draw(szNumber, rectAdj, crText, TI_TF_HALIGN_CENTER | TI_TF_SHADOW | TI_TF_VCENTER);
    }
    else
    {
      sprintf(szNumber, "%d", m_nStats[iStat]);
      font.Draw(szNumber, rectAdj, crText, TI_TF_HALIGN_CENTER | TI_TF_SHADOW | TI_TF_VCENTER);
    }
    

    rect += TiPoint(0, rect.Height());
    rectName += TiPoint(0, rect.Height());
    rectBase += TiPoint(0, rect.Height());
    rectAdj += TiPoint(0, rect.Height());
  }
}

//***********************************************************************
// ProcessKeyDown
//***********************************************************************

bool TiStatsWindow::ProcessKeyDown( int32 KeyCode )
{
  return( false );
}

//***********************************************************************
// ProcessButtonDown
//***********************************************************************

bool TiStatsWindow::ProcessButtonDown( int32 DeviceId, int32 ButtonId )
{
  if( DeviceId != m_iPlayer )
  {
    return( false ); 
  }

  switch( TiWindow::GetButton( ButtonId ) )
  {
  case TiButtonMap::kUp:
    Prev();
    return( true );

  case TiButtonMap::kDown:
    Next();
    return( true );

  case TiButtonMap::kLeft:
    UnassignPoint(m_iStat);
    return( true );

  case TiButtonMap::kRight:
    AssignPoint(m_iStat);
    return( true );

  case TiButtonMap::kX:
    CommitPoints();
    return( true );
    
  //case TiButtonMap::kCircle:
  //  RollbackPoints();
  //  return( true );
  
  default:
    return( false );
  }
  return( false );
}

//***********************************************************************
// ProcessReloadLayout
//***********************************************************************

void TiStatsWindow::ProcessReloadLayout()
{
}

void TiStatsWindow::Next()
{
  TiUI::Instance()->PlaySound(TiUI::kSoundTick);
  m_iStat = SY_MIN(m_iStat + 1, SW_MaxStats - 1);
}

void TiStatsWindow::Prev()
{
  TiUI::Instance()->PlaySound(TiUI::kSoundTick);
  m_iStat = SY_MAX(m_iStat - 1, 0);
}

void TiStatsWindow::AssignPoint(int iStat)
{
  if(!m_bLevelingUp || m_nBonusPoints == 0)
    return;
  TiUI::Instance()->PlaySound(TiUI::kSoundTick);
  m_nAdjustments[iStat]++;
  m_nBonusPoints--;
}

void TiStatsWindow::UnassignPoint(int iStat)
{
  if(!m_bLevelingUp || m_nAdjustments[iStat] == 0)
    return;
  TiUI::Instance()->PlaySound(TiUI::kSoundTick);
  m_nAdjustments[iStat]--;
  m_nBonusPoints++;
}

void TiStatsWindow::CommitPoints()
{
  if(!m_bLevelingUp)
    return;
  TiUI::Instance()->PlaySound(TiUI::kSoundTick);
  m_pData->GetTitanInterface()->AddHealthPoints(m_iPlayer, m_nAdjustments[SW_Health]);
  m_pData->GetTitanInterface()->AddManaPoints(m_iPlayer, m_nAdjustments[SW_Mana]);
  m_pData->GetTitanInterface()->AddAttackPowerPoints(m_iPlayer, m_nAdjustments[SW_AttackPower]);
  m_pData->GetTitanInterface()->AddMagicPowerPoints(m_iPlayer, m_nAdjustments[SW_MagicPower]);
  m_pData->GetTitanInterface()->AddMeleeDefensePoints(m_iPlayer, m_nAdjustments[SW_MeleeDefense]);
  m_pData->GetTitanInterface()->AddSpellDefensePoints(m_iPlayer, m_nAdjustments[SW_SpellDefense]);
  m_pData->Update();
  Init();
}

void TiStatsWindow::RollbackPoints()
{
  for(int i = 0; i < SW_MaxStats; i++)
  {
    m_nBonusPoints += m_nAdjustments[i];
    m_nAdjustments[i] = 0;
  }
}
