/********************************************************************

  Module: TiUI

  Author: Carlo Ordonez

  Description:

    Raster 2D drawing services

  Copyright 2005 Sony Online Entertainment.  All rights reserved.

*********************************************************************/

#ifndef _TiUI_h_
#define _TiUI_h_

#include "SyResourceID.h"
#include "TiBitmapInfo.h"
#include "SyMaterial.h"
#include "TitanI.h"
#include "TiPtSizeRect.h"
#include <map>

class SyRaster;
class SyDictionary;
class TiWindow;
class SyDeferred2D;
class SySoundDevice;


const SyColor32F c_crWhite(1.f, 1.f, 1.f, 1.f);
const SyColor32F c_crBlack(0.f, 0.f, 0.f, 1.f);
const SyColor32F c_crRed(1.f, 0.f, 0.f, 1.f);
const SyColor32F c_crGreen(0.f, 1.f, 0.f, 1.f);
const SyColor32F c_crBlue(0.f, 0.f, 1.f, 1.f);
const SyColor32F c_crItemEquipped(1.f, 250.f/255.f, 120.f/255.f, 1.f);
const SyColor32F c_crHalfAlpha(1.f, 1.f, 1.f, .5f);
const SyColor32F c_crFrameFill(34.f/255.f, 34.f/255.f, 34.f/255.f, 1.f);
const SyColor32F c_crMenuText(141.f/255.f, 148.f/255.f, 168.f/255.f, 1.f);

class TiUI : public TiUII
{
public:
  TiUI( SyRaster* pRaster, SySoundDevice* pSoundDevice, SyDictionary* pDictionary, std::map< SyResourceID, TiBitmapInfo >* pAtlasDirectory, int32 fontHandle );
  virtual ~TiUI();

  void      Reset();

  SyTime    GetTime();

#if 0           // old way, we now destroy and recreate this object on mode changes, so no need to refresh this asset handle - JEFFP
  void      SetFont(int32 fontHandle) { mFontHandle = fontHandle; }
#endif

  void      FlushDeferred2D();
  void      UpdateFrameElapse(SyTime time);
  float32   GetFrameElapse() const {return m_fFrameElapse;}

  int32             GetFontHandle(const char8 *pszFont) const;

  virtual int32     GetBitmapInfo( const char8* pFilename, SyVect2I& SurfaceOffset, SyVect2I& Size);
  virtual int32     GetBitmapInfo( const char8* pFilename, int32& SurfaceHandle, 
                                   SyVect2I& SurfaceOffset, SyVect2I& Size);
  virtual int32     GetBitmapInfo( uint32 uHashId, SyVect2I& SurfaceOffset, SyVect2I& Size);
  virtual int32     GetBitmapInfo( uint32 uHashId, int32& SurfaceHandle, 
                                   SyVect2I& SurfaceOffset, SyVect2I& Size);

  int32     GetFontHeight() const;
  int32     GetFontHeight(int32 nFontHandle) const;
  int32     GetStringWidth( const char8* pString );
  int32     GetStringWidth( const wchar_t* pString );

  void      GetDisplaySize( int32* pWidth, int32* pHeight ) const;
  void      SetDisplaySize( int32 Width, int32 Height );

  const SyVect2I GetCenter() const;

  int32     DrawString( const char8* pString, const SyVect2I& Location, const SyColor32F& Color, bool Shadow=false );
  int32     DrawString( const wchar_t* pString, const SyVect2I& Location, const SyColor32F& Color, bool Shadow=false );
  int32     DrawCenteredString( const char8* pString, const SyVect2I& Location, const SyColor32F& Color, uint32 CenterFlags, bool Shadow=false );
  int32     DrawCenteredString( const wchar_t* pString, const SyVect2I& Location, const SyColor32F& Color, uint32 CenterFlags, bool Shadow=false );

  void      FillRectangle( const SyVect2I& UL, const SyVect2I& LR, const SyColor32F& Color );
  void      FillRectangle( TiRect rect, const SyColor32F& crFill)
  {
    FillRectangle(SyVect2I(rect.left, rect.top), SyVect2I(rect.right, rect.bottom), crFill);
  }

  int32 Blit( 
    const char8* pFilename,
    uint32 BlitFlags,
    const SyVect2I& ScreenLoc,        
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 Blit( 
    const char8* pFilename, 
    uint32 BlitFlags,
    const SyVect2I& ScreenLoc, 
    const SyVect2I& SurfaceOffset, 
    const SyVect2I& ScreenSize,
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 Blit( 
    int32 SurfaceHandle, 
    uint32 BlitFlags,
    const SyVect2I& ScreenLoc, 
    const SyVect2I& SurfaceOffset, 
    const SyVect2I& ScreenSize,
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 StretchBlit( 
    const char8* pFilename, 
    uint32 BlitFlags,
    const SyVect2I& ScreenLoc, 
    const SyVect2I& ScreenSize,
    const SyVect2I& SurfaceOffset, 
    const SyVect2I& SurfaceSize,
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 StretchBlit( 
    int32 SurfaceHandle, 
    uint32 BlitFlags,
    const SyVect2I& ScreenLoc, 
    const SyVect2I& ScreenSize,
    const SyVect2I& SurfaceOffset, 
    const SyVect2I& SurfaceSize,
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 Blit( 
    const char8* pFilename,
    uint32 BlitFlags,
    const SyVect2& ScreenLoc,        
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 Blit( 
    const char8* pFilename, 
    uint32 BlitFlags,
    const SyVect2& ScreenLoc, 
    const SyVect2I& SurfaceOffset, 
    const SyVect2I& ScreenSize,
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 Blit( 
    int32 SurfaceHandle, 
    uint32 BlitFlags,
    const SyVect2& ScreenLoc, 
    const SyVect2I& SurfaceOffset, 
    const SyVect2I& ScreenSize,
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 StretchBlit( 
    const char8* pFilename, 
    uint32 BlitFlags,
    const SyVect2& ScreenLoc, 
    const SyVect2& ScreenSize,
    const SyVect2I& SurfaceOffset, 
    const SyVect2I& SurfaceSize,
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 StretchBlit( 
    int32 SurfaceHandle, 
    uint32 BlitFlags,
    const SyVect2& ScreenLoc, 
    const SyVect2& ScreenSize,
    const SyVect2I& SurfaceOffset, 
    const SyVect2I& SurfaceSize,
    SyAlphaBlendMode BlendMode = SYALPHABLENDMODE_AVG,
    float32 BlendAlpha = 1.f );

  int32 TriangleBlit(
      SyResourceID      idSurface,
      const TiPoint*    arScreenTriangle,
      const TiPoint*    arSurfaceTriangle,
      SyAlphaBlendMode  eBlendMode = SYALPHABLENDMODE_AVG,
      float32           fBlendAlpha = 1.f);

  enum Sound
  {
    kSoundBack,       // for returning to the previous screen
    kSoundCancel,     // for declining an option or to make a change
    kSoundEquip,      
    kSoundHealthUp,
    kSoundLevelUp,
    kSoundLootDrop,   // for a dying mob that drops loot items
    kSoundLowMana,
    kSoundName,       // for selecting something small like characters in a name (intermediate select)
    kSoundNegative,   // for when player selects an option that's unavailable
    kSoundNotice,     // for design-type messaging (e.g. when a friend appears online)
    kSoundPage,       // for paging between windows in the character inventory
    kSoundPickup,     // for confirmation when picking up a loot item
    kSoundSelect,     // for selecting something or a confirmation
    kSoundSpellSelect,
    kSoundTick,       // for moving between multiple choices (steps on a slider, letters on a keyboard, etc)
    kSoundUnequip,
    kSoundWarning,    // for a system-type warning (controller unplugged, network connection lost, etc)
    kSoundWindowEvoke,
    kSoundWindowClose,
  };

  int32 PlaySound( Sound sound );
  int32 PlaySound( const char8* pFilename );

  static TiUI           *Instance() {return sm_pSingleton;}
  static SyColor32F     ColorWithAlpha(SyColor32F crIn, float32 fAlpha) 
  {
    crIn.A *= fAlpha;
    return crIn;
  }

private:

  SyRaster*       mpRaster;
  SySoundDevice*  mpSoundDev;
  SyDictionary*   mpDictionary;
  int32           mFontHandle;
  SyVect2I        mDisplaySize;
  std::map< SyResourceID, TiBitmapInfo >* mpAtlasDirectory;

  SyDeferred2D*   mpDeferred2D;

  float32           m_fFrameElapse;
  static TiUI*      sm_pSingleton;
};

#endif

